/*
 * Decompiled with CFR 0.152.
 */
package io.jans.casa.plugins.certauthn.service;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.unboundid.ldap.sdk.Attribute;
import com.unboundid.ldap.sdk.DN;
import com.unboundid.ldap.sdk.RDN;
import io.jans.as.model.util.CertUtils;
import io.jans.casa.core.model.BasePerson;
import io.jans.casa.core.model.IdentityPerson;
import io.jans.casa.misc.Utils;
import io.jans.casa.plugins.certauthn.model.CertPerson;
import io.jans.casa.plugins.certauthn.model.Certificate;
import io.jans.casa.plugins.certauthn.service.PathCertificateVerifier;
import io.jans.casa.plugins.certauthn.service.UserCertificateMatch;
import io.jans.casa.service.IPersistenceService;
import io.jans.orm.search.filter.Filter;
import io.jans.scim.model.scim2.user.X509Certificate;
import java.io.ByteArrayInputStream;
import java.nio.charset.StandardCharsets;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateFactory;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import org.apache.commons.codec.digest.DigestUtils;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CertService {
    public static final String AGAMA_FLOW = "io.jans.casa.authn.cert";
    private static final String CERT_PREFIX = "cert:";
    private static final int DEFAULT_ROUND_TRIP_MAX_TIME = 30;
    private Logger logger = LoggerFactory.getLogger(this.getClass());
    private IPersistenceService persistenceService;
    private String certPickupUrl;
    private int roundTripMaxTime;
    private ObjectMapper mapper = new ObjectMapper();
    private boolean hasValidProperties;
    private List<java.security.cert.X509Certificate> certChain;
    private static CertService SINGLE_INSTANCE = null;

    private CertService() {
        this.persistenceService = (IPersistenceService)Utils.managedBean(IPersistenceService.class);
        this.reloadConfiguration();
    }

    public static CertService getInstance() {
        if (SINGLE_INSTANCE == null) {
            SINGLE_INSTANCE = new CertService();
        }
        return SINGLE_INSTANCE;
    }

    public boolean isHasValidProperties() {
        return this.hasValidProperties;
    }

    public void reloadConfiguration() {
        this.hasValidProperties = false;
        JSONObject props = Optional.ofNullable(this.persistenceService.getAgamaFlowConfigProperties(AGAMA_FLOW)).orElse(new JSONObject());
        String pemString = props.optString("certChainPEM", null);
        this.certPickupUrl = props.optString("certPickupUrl", null);
        if (pemString == null || this.certPickupUrl == null) {
            this.logger.error("Unable to read required config properties from flow {}", (Object)AGAMA_FLOW);
        } else {
            Integer rtmt = props.optIntegerObject("roundTripMaxTime", null);
            if (rtmt == null || rtmt < 10) {
                this.logger.warn("roundTripMaxTime provided is of no practical use. Setting a default value...");
                this.roundTripMaxTime = 30;
            } else {
                this.roundTripMaxTime = rtmt;
            }
            this.logger.info("Cert pickup URL will be: {}", (Object)this.certPickupUrl);
            this.logger.info("Round trip max time will be: {}", (Object)this.roundTripMaxTime);
            try (ByteArrayInputStream is = new ByteArrayInputStream(pemString.getBytes(StandardCharsets.UTF_8));){
                CertificateFactory cf = CertificateFactory.getInstance("X.509");
                this.certChain = cf.generateCertificates(is).stream().map(java.security.cert.X509Certificate.class::cast).collect(Collectors.toList());
                this.logger.info("{} certs loaded from certificate chain", (Object)this.certChain.size());
                this.hasValidProperties = true;
            }
            catch (Exception e) {
                this.logger.error(e.getMessage(), (Throwable)e);
            }
        }
    }

    public String getCertPickupUrl() {
        return this.certPickupUrl;
    }

    public int getRoundTripMaxTime() {
        return this.roundTripMaxTime;
    }

    public List<Certificate> getUserCerts(String userId) {
        List<Certificate> certs = new ArrayList<Certificate>();
        try {
            CertPerson person = (CertPerson)((Object)this.persistenceService.get(CertPerson.class, this.persistenceService.getPersonDn(userId)));
            List<X509Certificate> x509Certificates = this.getScimX509Certificates(Optional.ofNullable(person.getX509Certificates()).orElse(Collections.emptyList()));
            certs = person.getJansExtUid().stream().filter(uid -> uid.startsWith(CERT_PREFIX)).map(uid -> this.getExtraCertsInfo((String)uid, x509Certificates)).collect(Collectors.toList());
        }
        catch (Exception e) {
            this.logger.error(e.getMessage(), (Throwable)e);
        }
        return certs;
    }

    public int getDevicesTotal(String userId) {
        int total = 0;
        try {
            IdentityPerson person = (IdentityPerson)this.persistenceService.get(IdentityPerson.class, this.persistenceService.getPersonDn(userId));
            total = (int)person.getJansExtUid().stream().filter(uid -> uid.startsWith(CERT_PREFIX)).count();
        }
        catch (Exception e) {
            this.logger.error(e.getMessage(), (Throwable)e);
        }
        return total;
    }

    public UserCertificateMatch processMatch(java.security.cert.X509Certificate certificate, String userInum, boolean enroll) {
        UserCertificateMatch status = null;
        try {
            this.logger.info("Matching certificate and user. Enrollment is {}", (Object)enroll);
            CertPerson person = (CertPerson)((Object)this.persistenceService.get(CertPerson.class, this.persistenceService.getPersonDn(userInum)));
            if (person == null) {
                status = UserCertificateMatch.UNKNOWN_USER;
            } else {
                String externalUid = String.format("%s%s", CERT_PREFIX, this.getFingerPrint(certificate));
                Filter filter = Filter.createEqualityFilter((String)"jansExtUid", (Object)externalUid).multiValued();
                List people = this.persistenceService.find(BasePerson.class, this.persistenceService.getPeopleDn(), filter, 0, 1);
                if (people.size() > 0) {
                    status = userInum.equals(((BasePerson)people.get(0)).getInum()) ? (enroll ? UserCertificateMatch.CERT_ENROLLED_ALREADY : UserCertificateMatch.SUCCESS) : UserCertificateMatch.CERT_ENROLLED_OTHER_USER;
                } else if (enroll) {
                    this.logger.info("Associating presented cert to user");
                    ArrayList<String> oeuid = new ArrayList<String>(Optional.ofNullable(person.getJansExtUid()).orElse(Collections.emptyList()));
                    oeuid.add(externalUid);
                    person.setJansExtUid(oeuid);
                    this.updateUserX509Certificates(person, certificate);
                    status = this.persistenceService.modify((Object)person) ? UserCertificateMatch.SUCCESS : UserCertificateMatch.UNKNOWN_ERROR;
                } else {
                    this.logger.info("Certificate not associated to an existing account yet");
                    status = UserCertificateMatch.CERT_NOT_RECOGNIZED;
                }
            }
            this.logger.info("Operation result is {}", (Object)status.toString());
        }
        catch (Exception e) {
            this.logger.error(e.getMessage(), (Throwable)e);
            status = UserCertificateMatch.UNKNOWN_ERROR;
        }
        return status;
    }

    public boolean removeFromUser(String fingerPrint, String userId) throws Exception {
        int i;
        CertPerson person = (CertPerson)((Object)this.persistenceService.get(CertPerson.class, this.persistenceService.getPersonDn(userId)));
        List stringCerts = Optional.ofNullable(person.getX509Certificates()).orElse(new ArrayList());
        List<X509Certificate> scimCerts = this.getScimX509Certificates(stringCerts);
        boolean found = false;
        for (i = 0; i < scimCerts.size() && !found; ++i) {
            String val = scimCerts.get(i).getValue();
            found = this.getFingerPrint(CertUtils.x509CertificateFromPem((String)val)).equals(fingerPrint);
        }
        if (found) {
            this.logger.info("Removing cert from SCIM profile data");
            person.getX509Certificates().remove(i - 1);
        }
        person.getJansExtUid().remove(CERT_PREFIX + fingerPrint);
        this.logger.info("Removing cert reference from user");
        return this.persistenceService.modify((Object)person);
    }

    public boolean validate(java.security.cert.X509Certificate cert) {
        this.logger.info("Validating certificate...");
        PathCertificateVerifier verifier = new PathCertificateVerifier(true);
        return verifier.validate(cert, this.certChain);
    }

    private void updateUserX509Certificates(CertPerson person, java.security.cert.X509Certificate certificate) {
        List stringCerts = Optional.ofNullable(person.getX509Certificates()).orElse(new ArrayList());
        X509Certificate scimX509Cert = new X509Certificate();
        try {
            byte[] DEREncoded = certificate.getEncoded();
            scimX509Cert.setValue(new String(Base64.getEncoder().encode(DEREncoded), StandardCharsets.UTF_8));
            scimX509Cert.setDisplay(certificate.getSubjectX500Principal().getName());
            this.logger.debug("Updating user's jans509Certificate attribute");
            stringCerts.add(this.mapper.writeValueAsString((Object)scimX509Cert));
            person.setX509Certificates(stringCerts);
        }
        catch (Exception e) {
            this.logger.error(e.getMessage(), (Throwable)e);
        }
    }

    private Certificate getExtraCertsInfo(String externalUid, List<X509Certificate> scimCerts) {
        String fingerPrint = externalUid.replace(CERT_PREFIX, "");
        Certificate cert = new Certificate();
        cert.setFingerPrint(fingerPrint);
        for (X509Certificate sc : scimCerts) {
            try {
                java.security.cert.X509Certificate x509Certificate = CertUtils.x509CertificateFromPem((String)sc.getValue());
                if (!fingerPrint.equals(this.getFingerPrint(x509Certificate))) continue;
                Map<String, String> attributes = this.getDNAttributes(sc.getDisplay());
                String cn = attributes.get("cn");
                Object ou = attributes.getOrDefault("ou", "");
                String o = attributes.getOrDefault("o", "");
                if (Utils.isNotEmpty((String)ou)) {
                    if (Utils.isNotEmpty((String)o)) {
                        ou = (String)ou + ", " + o;
                    }
                } else if (Utils.isNotEmpty((String)o)) {
                    ou = o;
                }
                String l = attributes.getOrDefault("l", "");
                String st = attributes.getOrDefault("st", "");
                String c = attributes.getOrDefault("c", "");
                cert.setCommonName(cn);
                cert.setOrganization((String)ou);
                cert.setLocation(String.format("%s %s %s", l, st, c).trim());
                cert.setFormattedName(cn + (Utils.isEmpty((String)ou) ? "" : String.format(" (%s)", ou)));
                long date = x509Certificate.getNotAfter().getTime();
                cert.setExpirationDate(date);
                cert.setExpired(date < System.currentTimeMillis());
                break;
            }
            catch (Exception e) {
                this.logger.error(e.getMessage());
            }
        }
        return cert;
    }

    private List<X509Certificate> getScimX509Certificates(List<String> scimStringCerts) {
        ArrayList<X509Certificate> scimCerts = new ArrayList<X509Certificate>();
        for (String scimCert : scimStringCerts) {
            try {
                scimCerts.add((X509Certificate)this.mapper.readValue(scimCert, X509Certificate.class));
            }
            catch (Exception e) {
                this.logger.error("Unable to convert value '{}' to expected SCIM format", (Object)scimCert);
                this.logger.error(e.getMessage());
            }
        }
        return scimCerts;
    }

    private String getFingerPrint(java.security.cert.X509Certificate certificate) throws CertificateEncodingException {
        return DigestUtils.sha1Hex((byte[])certificate.getEncoded());
    }

    private Map<String, String> getDNAttributes(String dn) {
        HashMap<String, String> map = new HashMap<String, String>();
        if (Utils.isNotEmpty((String)dn)) {
            try {
                RDN[] rdns;
                for (RDN rdn : rdns = DN.getRDNs((String)dn)) {
                    Attribute[] attrs;
                    for (Attribute attr : attrs = rdn.getAttributes()) {
                        String name = attr.getName().toLowerCase();
                        if (map.containsKey(name)) continue;
                        map.put(name, attr.getValues()[0]);
                    }
                }
            }
            catch (Exception e) {
                this.logger.error(e.getMessage(), (Throwable)e);
            }
        }
        return map;
    }
}

