/*
 * Decompiled with CFR 0.152.
 */
package io.jans.as.server.auth;

import com.nimbusds.jose.jwk.JWKException;
import io.jans.as.common.model.registration.Client;
import io.jans.as.model.authorize.AuthorizeErrorResponseType;
import io.jans.as.model.configuration.AppConfiguration;
import io.jans.as.model.crypto.AbstractCryptoProvider;
import io.jans.as.model.error.ErrorResponseFactory;
import io.jans.as.model.error.IErrorType;
import io.jans.as.model.exception.CryptoProviderException;
import io.jans.as.model.exception.InvalidJwtException;
import io.jans.as.model.jwk.JSONWebKey;
import io.jans.as.model.jwk.JSONWebKeySet;
import io.jans.as.model.jwt.Jwt;
import io.jans.as.model.jwt.JwtType;
import io.jans.as.model.token.TokenErrorResponseType;
import io.jans.as.server.audit.ApplicationAuditLogger;
import io.jans.as.server.model.audit.OAuth2AuditLog;
import io.jans.as.server.model.common.DPoPJti;
import io.jans.as.server.util.ServerUtil;
import io.jans.service.CacheService;
import jakarta.ejb.DependsOn;
import jakarta.inject.Inject;
import jakarta.inject.Named;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.ws.rs.WebApplicationException;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.util.Date;
import java.util.UUID;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.json.JSONObject;
import org.slf4j.Logger;

@DependsOn(value={"appInitializer"})
@Named
public class DpopService {
    public static final String NO_CACHE = "no-cache";
    public static final String PRAGMA = "Pragma";
    public static final String DPOP_NONCE = "DPoP-Nonce";
    public static final String DPOP = "DPoP";
    @Inject
    private Logger log;
    @Inject
    private AppConfiguration appConfiguration;
    @Inject
    private CacheService cacheService;
    @Inject
    private AbstractCryptoProvider cryptoProvider;
    @Inject
    private ErrorResponseFactory errorResponseFactory;
    @Inject
    private ApplicationAuditLogger applicationAuditLogger;

    public void validateDpopValuesCount(HttpServletRequest servletRequest) {
        this.validateDpopValuesCount(servletRequest.getParameterValues(DPOP));
    }

    public void validateDpopValuesCount(String[] values) {
        if (values != null && values.length > 1) {
            this.log.trace("Multiple DPoP header values are not allowed. Count: {}", (Object)values.length);
            throw new WebApplicationException(Response.status((Response.Status)Response.Status.BAD_REQUEST).type(MediaType.APPLICATION_JSON_TYPE).entity((Object)this.errorResponseFactory.errorAsJson((IErrorType)TokenErrorResponseType.INVALID_DPOP_PROOF, "Multiple DPoP header values")).cacheControl(ServerUtil.cacheControl(true, false)).header(PRAGMA, (Object)NO_CACHE).build());
        }
    }

    public boolean validateDpop(String dpop) {
        try {
            return this.validateDpop(Jwt.parseOrThrow((String)dpop));
        }
        catch (InvalidJwtException e) {
            this.log.error("Failed to validate dpop: " + dpop, (Throwable)e);
            return false;
        }
    }

    public boolean validateDpop(Jwt dpop) {
        try {
            this.validateDpopHeader(dpop);
            this.validateDpopPayload(dpop);
            JSONWebKey jwk = JSONWebKey.fromJSONObject((JSONObject)dpop.getHeader().getJwk());
            String dpopJwkThumbprint = jwk.getJwkThumbprint();
            return this.validateDpopSignature(dpop, jwk, dpopJwkThumbprint);
        }
        catch (WebApplicationException e) {
            throw e;
        }
        catch (InvalidJwtException e) {
            this.log.error("Failed to validate dpop: " + dpop, (Throwable)e);
        }
        catch (Exception e) {
            this.log.error("Invalid dpop: " + dpop, (Throwable)e);
        }
        return false;
    }

    private boolean validateDpopSignature(Jwt dpop, JSONWebKey jwk, String dpopJwkThumbprint) throws InvalidJwtException, CryptoProviderException {
        if (dpopJwkThumbprint == null) {
            throw new InvalidJwtException("Invalid DPoP Proof Header. The jwk header is not valid.");
        }
        JSONWebKeySet jwks = new JSONWebKeySet();
        jwks.getKeys().add(jwk);
        return this.cryptoProvider.verifySignature(dpop.getSigningInput(), dpop.getEncodedSignature(), null, jwks.toJSONObject(), null, dpop.getHeader().getSignatureAlgorithm());
    }

    private void validateDpopPayload(Jwt dpop) throws InvalidJwtException {
        if (StringUtils.isBlank((CharSequence)dpop.getClaims().getClaimAsString("htm"))) {
            throw new InvalidJwtException("Invalid DPoP Proof Payload. The htm param is required.");
        }
        if (StringUtils.isBlank((CharSequence)dpop.getClaims().getClaimAsString("htu"))) {
            throw new InvalidJwtException("Invalid DPoP Proof Payload. The htu param is required");
        }
        if (dpop.getClaims().getClaimAsLong("iat") == null) {
            throw new InvalidJwtException("Invalid DPoP Proof Payload. The iat param is required.");
        }
        if (StringUtils.isBlank((CharSequence)dpop.getClaims().getClaimAsString("jti"))) {
            throw new InvalidJwtException("Invalid DPoP Proof Payload. The jti param is required");
        }
        String jti = dpop.getClaims().getClaimAsString("jti");
        Long iat = dpop.getClaims().getClaimAsLong("iat");
        String htu = dpop.getClaims().getClaimAsString("htu");
        String nonce = dpop.getClaims().getClaimAsString("nonce");
        String cacheKey = "dpop_jti_" + jti;
        DPoPJti dPoPJti = (DPoPJti)this.cacheService.get(cacheKey);
        int seconds = this.appConfiguration.getDpopTimeframe();
        long diff = (new Date().getTime() - iat) / 1000L;
        if (diff > (long)seconds) {
            throw new InvalidJwtException("The DPoP token has expired.");
        }
        if (dPoPJti != null) {
            throw new InvalidJwtException("Invalid DPoP Proof. The jti param has been used before.");
        }
        dPoPJti = new DPoPJti(jti, iat, htu);
        this.cacheService.put(this.appConfiguration.getDpopJtiCacheTime(), cacheKey, (Object)dPoPJti);
        this.validateDpopNonce(nonce);
    }

    public void validateDpopNonce(String nonce) {
        if (BooleanUtils.isFalse((Boolean)this.appConfiguration.getDpopUseNonce())) {
            return;
        }
        if (StringUtils.isBlank((CharSequence)nonce)) {
            throw new WebApplicationException(Response.status((Response.Status)Response.Status.BAD_REQUEST).type(MediaType.APPLICATION_JSON_TYPE).entity((Object)this.errorResponseFactory.errorAsJson((IErrorType)TokenErrorResponseType.USE_DPOP_NONCE, "Nonce is not set")).cacheControl(ServerUtil.cacheControl(true, false)).header(PRAGMA, (Object)NO_CACHE).header(DPOP_NONCE, (Object)this.generateDpopNonce()).build());
        }
        Object nonceValue = this.cacheService.get(nonce);
        if (nonceValue == null) {
            throw new WebApplicationException(Response.status((Response.Status)Response.Status.BAD_REQUEST).type(MediaType.APPLICATION_JSON_TYPE).entity((Object)this.errorResponseFactory.errorAsJson((IErrorType)TokenErrorResponseType.USE_NEW_DPOP_NONCE, "New nonce value is required")).cacheControl(ServerUtil.cacheControl(true, false)).header(PRAGMA, (Object)NO_CACHE).header(DPOP_NONCE, (Object)this.generateDpopNonce()).build());
        }
    }

    private String generateDpopNonce() {
        String nonce = UUID.randomUUID().toString();
        this.cacheService.put(this.appConfiguration.getDpopNonceCacheTime(), nonce, (Object)nonce);
        return nonce;
    }

    private void validateDpopHeader(Jwt dpop) throws InvalidJwtException {
        if (dpop.getHeader().getType() != JwtType.DPOP_PLUS_JWT) {
            throw new InvalidJwtException("Invalid DPoP Proof Header. The typ header must be dpop+jwt.");
        }
        if (dpop.getHeader().getSignatureAlgorithm() == null) {
            throw new InvalidJwtException("Invalid DPoP Proof Header. The typ header must be dpop+jwt.");
        }
        if (dpop.getHeader().getJwk() == null) {
            throw new InvalidJwtException("Invalid DPoP Proof Header. The jwk header is required.");
        }
    }

    private Response response(Response.ResponseBuilder builder, OAuth2AuditLog oAuth2AuditLog) {
        builder.cacheControl(ServerUtil.cacheControl(true, false));
        builder.header(PRAGMA, (Object)NO_CACHE);
        this.applicationAuditLogger.sendMessage(oAuth2AuditLog);
        return builder.build();
    }

    private Response.ResponseBuilder error(int status, TokenErrorResponseType type, String reason) {
        return Response.status((int)status).type(MediaType.APPLICATION_JSON_TYPE).entity((Object)this.errorResponseFactory.errorAsJson((IErrorType)type, reason));
    }

    public static String getDpopJwkThumbprint(String dpopStr) throws InvalidJwtException, NoSuchAlgorithmException, JWKException, NoSuchProviderException {
        Jwt dpop = Jwt.parseOrThrow((String)dpopStr);
        JSONWebKey jwk = JSONWebKey.fromJSONObject((JSONObject)dpop.getHeader().getJwk());
        return jwk.getJwkThumbprint();
    }

    public String getDPoPJwkThumbprint(HttpServletRequest httpRequest, Client client, OAuth2AuditLog oAuth2AuditLog) {
        try {
            String dpopStr = httpRequest.getHeader(DPOP);
            boolean isDpopBlank = StringUtils.isBlank((CharSequence)dpopStr);
            if (BooleanUtils.isTrue((Boolean)client.getAttributes().getDpopBoundAccessToken()) && isDpopBlank) {
                this.log.debug("Client requires DPoP bound access token. Invalid request - DPoP header is not set.");
                throw new WebApplicationException(this.response(this.error(400, TokenErrorResponseType.INVALID_DPOP_PROOF, "Invalid request - DPoP header is not set."), oAuth2AuditLog));
            }
            if (isDpopBlank) {
                return null;
            }
            String dpopJwkThumbprint = DpopService.getDpopJwkThumbprint(dpopStr);
            if (dpopJwkThumbprint == null) {
                throw new InvalidJwtException("Invalid DPoP Proof Header. The jwk header is not valid.");
            }
            return dpopJwkThumbprint;
        }
        catch (JWKException | InvalidJwtException | NoSuchAlgorithmException | NoSuchProviderException e) {
            throw new WebApplicationException(this.response(this.error(400, TokenErrorResponseType.INVALID_DPOP_PROOF, e.getMessage()), oAuth2AuditLog));
        }
    }

    public void validateDpopThumprintIsPresent(String dpopJkt, String state) {
        if (BooleanUtils.isTrue((Boolean)this.appConfiguration.getDpopJktForceForAuthorizationCode()) && StringUtils.isBlank((CharSequence)dpopJkt)) {
            throw new WebApplicationException(Response.status((int)Response.Status.BAD_REQUEST.getStatusCode()).type(MediaType.APPLICATION_JSON_TYPE).entity((Object)this.errorResponseFactory.getErrorAsJson((IErrorType)AuthorizeErrorResponseType.INVALID_REQUEST, state, "dpop_jkt is absent")).build());
        }
    }

    public void validateDpopThumprint(String existingThumprint, String requestThumprint) {
        if (StringUtils.isBlank((CharSequence)existingThumprint) && BooleanUtils.isFalse((Boolean)this.appConfiguration.getDpopJktForceForAuthorizationCode())) {
            return;
        }
        if (!StringUtils.equals((CharSequence)existingThumprint, (CharSequence)requestThumprint)) {
            this.log.debug("DPoP Thumprint between saved one '{}' and send in request '{}' does NOT match. Reject request.", (Object)existingThumprint, (Object)requestThumprint);
            throw new WebApplicationException(Response.status((Response.Status)Response.Status.BAD_REQUEST).type(MediaType.APPLICATION_JSON_TYPE).entity((Object)this.errorResponseFactory.errorAsJson((IErrorType)TokenErrorResponseType.INVALID_DPOP_PROOF, "Thumprint does not match")).cacheControl(ServerUtil.cacheControl(true, false)).header(PRAGMA, (Object)NO_CACHE).build());
        }
    }
}

