/*
 * Decompiled with CFR 0.152.
 */
package io.jans.as.server.service;

import com.google.common.collect.Lists;
import io.jans.as.model.config.StaticConfiguration;
import io.jans.as.model.configuration.AppConfiguration;
import io.jans.as.model.configuration.LockMessageConfig;
import io.jans.as.server.model.common.AuthorizationGrant;
import io.jans.as.server.model.common.CacheGrant;
import io.jans.as.server.service.ClientService;
import io.jans.as.server.service.token.StatusListIndexService;
import io.jans.as.server.util.TokenHashUtil;
import io.jans.model.token.TokenEntity;
import io.jans.model.token.TokenType;
import io.jans.model.tokenstatus.TokenStatus;
import io.jans.orm.PersistenceEntryManager;
import io.jans.orm.search.filter.Filter;
import io.jans.service.CacheService;
import io.jans.service.MessageService;
import io.jans.util.StringHelper;
import jakarta.enterprise.context.ApplicationScoped;
import jakarta.inject.Inject;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;

@ApplicationScoped
public class GrantService {
    private static final ExecutorService statusListPool = Executors.newFixedThreadPool(5, runnable -> {
        Thread thread = new Thread(runnable);
        thread.setName("grant_service_status_list_pool");
        thread.setDaemon(true);
        return thread;
    });
    @Inject
    private Logger log;
    @Inject
    private PersistenceEntryManager persistenceEntryManager;
    @Inject
    private ClientService clientService;
    @Inject
    private MessageService messageService;
    @Inject
    private CacheService cacheService;
    @Inject
    private StaticConfiguration staticConfiguration;
    @Inject
    private AppConfiguration appConfiguration;
    @Inject
    private StatusListIndexService statusListIndexService;

    public static String generateGrantId() {
        return UUID.randomUUID().toString();
    }

    public String buildDn(String hashedToken) {
        return String.format("tknCde=%s,", hashedToken) + this.tokenBaseDn();
    }

    private String tokenBaseDn() {
        return this.staticConfiguration.getBaseDn().getTokens();
    }

    public void merge(TokenEntity token) {
        this.persistenceEntryManager.merge((Object)token);
    }

    public void mergeSilently(TokenEntity token) {
        try {
            this.persistenceEntryManager.merge((Object)token);
        }
        catch (Exception e) {
            this.log.error(e.getMessage(), (Throwable)e);
        }
    }

    public boolean shouldPersist() {
        return !BooleanUtils.isTrue((Boolean)this.appConfiguration.getSaveTokensInCacheAndDontSaveInPersistence());
    }

    public boolean shouldSaveInCache() {
        return BooleanUtils.isTrue((Boolean)this.appConfiguration.getSaveTokensInCache()) || BooleanUtils.isTrue((Boolean)this.appConfiguration.getSaveTokensInCacheAndDontSaveInPersistence());
    }

    public void persist(TokenEntity token) {
        if (token.isAccessToken() || token.isLogoutStatusJwt() || this.shouldPersist()) {
            this.persistenceEntryManager.persist((Object)token);
        }
        if (this.shouldSaveInCache()) {
            this.saveInCache(token);
        }
        if (TokenType.ACCESS_TOKEN.getValue().equals(token.getTokenType())) {
            this.publishIdTokenLockMessage(token, "add");
        }
    }

    private void saveInCache(TokenEntity token) {
        long lifeTimeAsMillis = token.getExpirationDate().getTime() - System.currentTimeMillis();
        int lifetimeInSeconds = (int)(lifeTimeAsMillis / 1000L);
        this.cacheService.put(lifetimeInSeconds, TokenHashUtil.hash(token.getTokenCode()), (Object)token.getTokenCode());
    }

    public void remove(TokenEntity token) {
        this.persistenceEntryManager.remove((Object)token);
        this.log.trace("Removed token from DB, code: {}", (Object)token.getTokenCode());
        if (TokenType.ACCESS_TOKEN == token.getTokenTypeEnum()) {
            this.publishIdTokenLockMessage(token, "del");
        }
    }

    protected void publishIdTokenLockMessage(TokenEntity token, String opearation) {
        LockMessageConfig lockMessageConfig = this.appConfiguration.getLockMessageConfig();
        if (lockMessageConfig == null) {
            return;
        }
        if (Boolean.TRUE.equals(lockMessageConfig.getEnableTokenMessages()) && StringHelper.isNotEmpty((String)lockMessageConfig.getTokenMessagesChannel())) {
            String jsonMessage = String.format("{\"tknTyp\" : \"%s\", \"tknId\" : \"%s\", \"tknOp\" : \"%s\"}", token.getTokenType(), token.getTokenCode(), opearation);
            this.messageService.publish(lockMessageConfig.getTokenMessagesChannel(), jsonMessage);
        }
    }

    public void removeSilently(TokenEntity token) {
        try {
            this.remove(token);
            if (StringUtils.isNotBlank((CharSequence)token.getAuthorizationCode())) {
                this.cacheService.remove(CacheGrant.cacheKey(token.getAuthorizationCode(), token.getGrantId()));
            }
            if (this.shouldSaveInCache()) {
                this.cacheService.remove(token.getTokenCode());
            }
            statusListPool.execute(() -> {
                Integer index = token.getAttributes().getStatusListIndex();
                if (index != null && index > 0) {
                    this.statusListIndexService.updateStatusAtIndexes(Lists.newArrayList((Object[])new Integer[]{index}), TokenStatus.INVALID);
                }
            });
        }
        catch (Exception e) {
            this.log.error(e.getMessage(), (Throwable)e);
        }
    }

    public void remove(List<TokenEntity> entries) {
        if (entries != null && !entries.isEmpty()) {
            for (TokenEntity t : entries) {
                try {
                    this.remove(t);
                }
                catch (Exception e) {
                    this.log.error("Failed to remove entry", (Throwable)e);
                }
            }
        }
    }

    public void removeSilently(List<TokenEntity> entries) {
        if (entries != null && !entries.isEmpty()) {
            ArrayList<Integer> indexes = new ArrayList<Integer>();
            for (TokenEntity t : entries) {
                try {
                    Integer index;
                    this.remove(t);
                    if (StringUtils.isNotBlank((CharSequence)t.getAuthorizationCode())) {
                        this.cacheService.remove(CacheGrant.cacheKey(t.getAuthorizationCode(), t.getGrantId()));
                    }
                    if (this.shouldSaveInCache()) {
                        this.cacheService.remove(t.getTokenCode());
                    }
                    if ((index = t.getAttributes().getStatusListIndex()) == null || index < 0) continue;
                    indexes.add(index);
                }
                catch (Exception e) {
                    this.log.error(e.getMessage(), (Throwable)e);
                }
            }
            statusListPool.execute(() -> this.statusListIndexService.updateStatusAtIndexes(indexes, TokenStatus.INVALID));
        }
    }

    public void remove(AuthorizationGrant grant) {
        if (grant != null && grant.getTokenEntity() != null) {
            try {
                this.remove(grant.getTokenEntity());
            }
            catch (Exception e) {
                this.log.error(e.getMessage(), (Throwable)e);
            }
        }
    }

    public List<TokenEntity> getGrantsOfClient(String clientId) {
        try {
            String baseDn = this.clientService.buildClientDn(clientId);
            return this.persistenceEntryManager.findEntries(baseDn, TokenEntity.class, Filter.createPresenceFilter((String)"tknCde"));
        }
        catch (Exception e) {
            this.logException(e);
            return Collections.emptyList();
        }
    }

    public TokenEntity getGrantByCode(String code) {
        Object grant = this.cacheService.get(TokenHashUtil.hash(code));
        if (grant instanceof TokenEntity) {
            return (TokenEntity)grant;
        }
        return this.load(this.buildDn(TokenHashUtil.hash(code)));
    }

    public TokenEntity getGrantByReferenceId(String referenceId) {
        try {
            List grants = this.persistenceEntryManager.findEntries(this.tokenBaseDn(), TokenEntity.class, Filter.createEqualityFilter((String)"jansId", (Object)referenceId));
            if (grants.size() > 1) {
                this.log.error("Found more then one tokens by referenceId {}", (Object)referenceId);
                return null;
            }
            if (grants.size() == 1) {
                return (TokenEntity)grants.get(0);
            }
        }
        catch (Exception e) {
            this.logException(e);
        }
        return null;
    }

    private void logException(Exception e) {
        if (BooleanUtils.isTrue((Boolean)this.appConfiguration.getLogNotFoundEntityAsError())) {
            this.log.error(e.getMessage(), (Throwable)e);
        } else {
            this.log.trace(e.getMessage(), (Throwable)e);
        }
    }

    private TokenEntity load(String tokenDn) {
        try {
            return (TokenEntity)this.persistenceEntryManager.find(TokenEntity.class, (Object)tokenDn);
        }
        catch (Exception e) {
            this.logException(e);
            return null;
        }
    }

    public TokenEntity getGrantsByJti(String jti) {
        try {
            List grants = this.persistenceEntryManager.findEntries(this.tokenBaseDn(), TokenEntity.class, Filter.createEqualityFilter((String)"jti", (Object)jti));
            if (grants.size() > 1) {
                this.log.error("Found more then one tokens by jti {}", (Object)jti);
                return null;
            }
            if (grants.size() == 1) {
                return (TokenEntity)grants.get(0);
            }
        }
        catch (Exception e) {
            this.logException(e);
        }
        return null;
    }

    public List<TokenEntity> getGrantsByGrantId(String grantId) {
        try {
            return this.persistenceEntryManager.findEntries(this.tokenBaseDn(), TokenEntity.class, Filter.createEqualityFilter((String)"grtId", (Object)grantId));
        }
        catch (Exception e) {
            this.logException(e);
            return Collections.emptyList();
        }
    }

    public List<TokenEntity> getGrantsByAuthorizationCode(String authorizationCode) {
        try {
            return this.persistenceEntryManager.findEntries(this.tokenBaseDn(), TokenEntity.class, Filter.createEqualityFilter((String)"authzCode", (Object)TokenHashUtil.hash(authorizationCode)));
        }
        catch (Exception e) {
            this.logException(e);
            return Collections.emptyList();
        }
    }

    public List<TokenEntity> getGrantsBySessionDn(String sessionDn) {
        ArrayList<TokenEntity> grants = new ArrayList<TokenEntity>();
        try {
            List ldapGrants = this.persistenceEntryManager.findEntries(this.tokenBaseDn(), TokenEntity.class, Filter.createEqualityFilter((String)"ssnId", (Object)sessionDn));
            if (ldapGrants != null) {
                grants.addAll(ldapGrants);
            }
        }
        catch (Exception e) {
            this.logException(e);
        }
        return grants;
    }

    public List<TokenEntity> getGrantsByUserDn(String userDn) {
        ArrayList<TokenEntity> grants = new ArrayList<TokenEntity>();
        try {
            List tokenEntities = this.persistenceEntryManager.findEntries(this.tokenBaseDn(), TokenEntity.class, Filter.createEqualityFilter((String)"jansUsrDN", (Object)userDn));
            if (tokenEntities != null) {
                grants.addAll(tokenEntities);
            }
        }
        catch (Exception e) {
            this.logException(e);
        }
        return grants;
    }

    public void logout(String sessionDn) {
        List<TokenEntity> tokens = this.getGrantsBySessionDn(sessionDn);
        this.filterOutRefreshTokenFromDeletion(tokens);
        this.removeSilently(tokens);
    }

    public void filterOutRefreshTokenFromDeletion(List<TokenEntity> tokens) {
        if (BooleanUtils.isTrue((Boolean)this.appConfiguration.getRemoveRefreshTokensForClientOnLogout())) {
            return;
        }
        ArrayList refreshTokensForExclusion = Lists.newArrayList();
        for (TokenEntity token : tokens) {
            if (token.getTokenTypeEnum() != TokenType.REFRESH_TOKEN || token.getAttributes().isOnlineAccess()) continue;
            refreshTokensForExclusion.add(token);
        }
        if (!refreshTokensForExclusion.isEmpty()) {
            this.log.trace("Refresh tokens are not removed on logout (because removeRefreshTokensForClientOnLogout configuration property is false or online_access scope is used).");
            tokens.removeAll(refreshTokensForExclusion);
        }
    }

    public void removeAllTokensBySession(String sessionDn) {
        this.removeSilently(this.getGrantsBySessionDn(sessionDn));
    }

    public void removeByCode(String code) {
        TokenEntity t = this.getGrantByCode(code);
        if (t != null) {
            this.removeSilently(t);
        }
        this.cacheService.remove(CacheGrant.cacheKey(code, null));
    }

    public void removeAuthorizationCode(String code) {
        this.cacheService.remove(CacheGrant.cacheKey(code, null));
    }

    public void removeAllByAuthorizationCode(String authorizationCode) {
        this.removeSilently(this.getGrantsByAuthorizationCode(authorizationCode));
    }

    public void removeAllByGrantId(String grantId) {
        this.removeSilently(this.getGrantsByGrantId(grantId));
    }
}

