/*
 * Decompiled with CFR 0.152.
 */
package io.jans.as.server.service;

import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import io.jans.as.common.model.registration.Client;
import io.jans.as.model.common.AuthenticationMethod;
import io.jans.as.model.config.StaticConfiguration;
import io.jans.as.model.configuration.AppConfiguration;
import io.jans.as.persistence.model.Scope;
import io.jans.as.server.model.token.HandleTokenFactory;
import io.jans.as.server.service.ClientLastUpdateAtTimer;
import io.jans.as.server.service.ScopeService;
import io.jans.orm.PersistenceEntryManager;
import io.jans.orm.model.base.CustomObjectAttribute;
import io.jans.service.BaseCacheService;
import io.jans.service.CacheService;
import io.jans.service.EncryptionService;
import io.jans.service.LocalCacheService;
import io.jans.util.StringHelper;
import io.jans.util.security.StringEncrypter;
import jakarta.enterprise.context.ApplicationScoped;
import jakarta.inject.Inject;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.apache.commons.lang3.BooleanUtils;
import org.json.JSONArray;
import org.slf4j.Logger;

@ApplicationScoped
public class ClientService {
    @Inject
    private Logger log;
    @Inject
    private PersistenceEntryManager persistenceEntryManager;
    @Inject
    private CacheService cacheService;
    @Inject
    private LocalCacheService localCacheService;
    @Inject
    private ScopeService scopeService;
    @Inject
    private EncryptionService encryptionService;
    @Inject
    private AppConfiguration appConfiguration;
    @Inject
    private StaticConfiguration staticConfiguration;
    @Inject
    private ClientLastUpdateAtTimer clientLastUpdateAtTimer;

    public void persist(Client client) {
        this.ignoreCustomObjectClassesForNonLDAP(client);
        this.persistenceEntryManager.persist((Object)client);
    }

    private Client ignoreCustomObjectClassesForNonLDAP(Client client) {
        String persistenceType = this.persistenceEntryManager.getPersistenceType();
        this.log.debug("persistenceType: {}", (Object)persistenceType);
        if (!PersistenceEntryManager.PERSITENCE_TYPES.ldap.name().equals(persistenceType)) {
            this.log.debug("Setting CustomObjectClasses :{} to null as it's used only for LDAP and current persistenceType is {} ", (Object)client.getCustomObjectClasses(), (Object)persistenceType);
            client.setCustomObjectClasses(null);
        }
        return client;
    }

    public void merge(Client client) {
        this.ignoreCustomObjectClassesForNonLDAP(client);
        this.persistenceEntryManager.merge((Object)client);
        this.removeFromCache(client);
    }

    public boolean authenticate(String clientId, String password) {
        this.log.debug("Authenticating Client with LDAP: clientId = {}", (Object)clientId);
        boolean authenticated = false;
        try {
            Client client = this.getClient(clientId);
            if (client == null) {
                this.log.debug("Failed to find client = {}", (Object)clientId);
                return authenticated;
            }
            String decryptedClientSecret = this.decryptSecret(client.getClientSecret());
            authenticated = decryptedClientSecret != null && decryptedClientSecret.equals(password);
        }
        catch (StringEncrypter.EncryptionException e) {
            this.log.error(e.getMessage(), (Throwable)e);
        }
        return authenticated;
    }

    public Set<Client> getClient(Collection<String> clientIds, boolean silent) {
        HashSet set = Sets.newHashSet();
        if (clientIds == null) {
            return set;
        }
        for (String clientId : clientIds) {
            try {
                Client client = this.getClient(clientId);
                if (client == null) continue;
                set.add(client);
            }
            catch (RuntimeException e) {
                if (silent) continue;
                throw e;
            }
        }
        return set;
    }

    public Client getClient(String clientId) {
        if (clientId != null && !clientId.isEmpty()) {
            Client result = this.getClientByDn(this.buildClientDn(clientId));
            this.log.debug("Found {} entries for client id = {}", (Object)(result != null ? 1 : 0), (Object)clientId);
            return result;
        }
        return null;
    }

    public boolean isPublic(String clientId) {
        return this.isPublic(this.getClient(clientId));
    }

    public boolean isPublic(Client client) {
        return client != null && client.hasAuthenticationMethod(AuthenticationMethod.NONE);
    }

    public Client getClient(String clientId, String registrationAccessToken) {
        Client client = this.getClient(clientId);
        if (client != null && registrationAccessToken != null && registrationAccessToken.equals(client.getRegistrationAccessToken())) {
            this.rotateRegistrationAccessToken(client);
            return client;
        }
        return null;
    }

    public String generateRegistrationAccessToken() {
        return HandleTokenFactory.generateHandleToken();
    }

    public void rotateRegistrationAccessToken(Client client) {
        if (client == null) {
            return;
        }
        if (BooleanUtils.isFalse((Boolean)this.appConfiguration.getRotateClientRegistrationAccessTokenOnUsage())) {
            return;
        }
        client.setRegistrationAccessToken(this.generateRegistrationAccessToken());
        this.persist(client);
    }

    public Set<Client> getClientsByDns(Collection<String> dnList) {
        return this.getClientsByDns(dnList, true);
    }

    public Set<Client> getClientsByDns(Collection<String> dnList, boolean silently) {
        Preconditions.checkNotNull(dnList);
        HashSet result = Sets.newHashSet();
        for (String clientDn : dnList) {
            try {
                result.add(this.getClientByDn(clientDn));
            }
            catch (RuntimeException e) {
                if (silently) continue;
                throw e;
            }
        }
        return result;
    }

    public Client getClientByDn(String dn) {
        BaseCacheService usedCacheService = this.getCacheService();
        try {
            return (Client)usedCacheService.getWithPut(dn, () -> (Client)this.persistenceEntryManager.find(Client.class, (Object)dn), 60);
        }
        catch (Exception e) {
            this.log.trace(e.getMessage(), (Throwable)e);
            return null;
        }
    }

    public CustomObjectAttribute getCustomAttribute(Client client, String attributeName) {
        for (CustomObjectAttribute customAttribute : client.getCustomAttributes()) {
            if (!StringHelper.equalsIgnoreCase((String)attributeName, (String)customAttribute.getName())) continue;
            return customAttribute;
        }
        return null;
    }

    public void setCustomAttribute(Client client, String attributeName, String attributeValue) {
        CustomObjectAttribute customAttribute = this.getCustomAttribute(client, attributeName);
        if (customAttribute == null) {
            customAttribute = new CustomObjectAttribute(attributeName);
            client.getCustomAttributes().add(customAttribute);
        }
        customAttribute.setValue((Object)attributeValue);
    }

    public List<Client> getAllClients(String[] returnAttributes) {
        String baseDn = this.staticConfiguration.getBaseDn().getClients();
        return this.persistenceEntryManager.findEntries(baseDn, Client.class, null, returnAttributes);
    }

    public List<Client> getAllClients(String[] returnAttributes, int size) {
        String baseDn = this.staticConfiguration.getBaseDn().getClients();
        return this.persistenceEntryManager.findEntries(baseDn, Client.class, null, returnAttributes, size);
    }

    public String buildClientDn(String clientId) {
        StringBuilder dn = new StringBuilder();
        dn.append(String.format("inum=%s,", clientId));
        dn.append(this.staticConfiguration.getBaseDn().getClients());
        return dn.toString();
    }

    public void remove(Client client) {
        if (client != null) {
            this.removeFromCache(client);
            String clientDn = client.getDn();
            this.persistenceEntryManager.removeRecursively(clientDn, Client.class);
        }
    }

    public void removeFromCache(Client client) {
        BaseCacheService usedCacheService = this.getCacheService();
        try {
            usedCacheService.remove(client.getDn());
        }
        catch (Exception e) {
            this.log.error("Failed to remove client from cache." + client.getDn(), (Throwable)e);
        }
    }

    public void updateAccessTime(Client client, boolean isUpdateLogonTime) {
        this.clientLastUpdateAtTimer.addLastUpdateAtTime(client, isUpdateLogonTime);
    }

    public Object getAttribute(Client client, String clientAttribute) {
        String attribute = null;
        if (clientAttribute != null) {
            if (clientAttribute.equals("displayName")) {
                attribute = client.getClientName();
            } else if (clientAttribute.equals("inum")) {
                attribute = client.getClientId();
            } else if (clientAttribute.equals("jansAppTyp")) {
                attribute = client.getApplicationType();
            } else if (clientAttribute.equals("jansIdTknSignedRespAlg")) {
                attribute = client.getIdTokenSignedResponseAlg();
            } else if (clientAttribute.equals("jansRedirectURI") && client.getRedirectUris() != null) {
                JSONArray array = new JSONArray();
                for (String redirectUri : client.getRedirectUris()) {
                    array.put((Object)redirectUri);
                }
                attribute = array;
            } else if (clientAttribute.equals("jansScope") && client.getScopes() != null) {
                JSONArray array = new JSONArray();
                for (String scopeDN : client.getScopes()) {
                    Scope s = this.scopeService.getScopeByDn(scopeDN);
                    if (s == null) continue;
                    String scopeName = s.getId();
                    array.put((Object)scopeName);
                }
                attribute = array;
            } else {
                for (CustomObjectAttribute customAttribute : client.getCustomAttributes()) {
                    if (!customAttribute.getName().equals(clientAttribute)) continue;
                    List values = customAttribute.getValues();
                    if (values == null) break;
                    if (values.size() == 1) {
                        attribute = values.get(0);
                        break;
                    }
                    JSONArray array = new JSONArray();
                    for (Object v : values) {
                        array.put(v);
                    }
                    attribute = array;
                    break;
                }
            }
        }
        return attribute;
    }

    public String decryptSecret(String encryptedClientSecret) throws StringEncrypter.EncryptionException {
        return this.encryptionService.decrypt(encryptedClientSecret);
    }

    public String encryptSecret(String clientSecret) throws StringEncrypter.EncryptionException {
        return this.encryptionService.encrypt(clientSecret);
    }

    private BaseCacheService getCacheService() {
        if (BooleanUtils.isTrue((Boolean)this.appConfiguration.getUseLocalCache())) {
            return this.localCacheService;
        }
        return this.cacheService;
    }
}

