/*
 * Decompiled with CFR 0.152.
 */
package io.jans.lock.cedarling.service.policy;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.jans.lock.cedarling.service.event.PolicyDownloadEvent;
import io.jans.lock.model.config.AppConfiguration;
import io.jans.lock.model.config.cedarling.CedarlingConfiguration;
import io.jans.lock.model.config.cedarling.PolicySource;
import io.jans.service.EncryptionService;
import io.jans.service.cdi.async.Asynchronous;
import io.jans.service.cdi.event.ApplicationInitialized;
import io.jans.service.cdi.event.ApplicationInitializedEvent;
import io.jans.service.cdi.event.Scheduled;
import io.jans.service.net.BaseHttpService;
import io.jans.service.timer.event.TimerEvent;
import io.jans.service.timer.schedule.TimerSchedule;
import io.jans.util.StringHelper;
import jakarta.annotation.PostConstruct;
import jakarta.enterprise.context.ApplicationScoped;
import jakarta.enterprise.event.Event;
import jakarta.enterprise.event.Observes;
import jakarta.inject.Inject;
import jakarta.ws.rs.core.MediaType;
import java.io.IOException;
import java.io.InputStream;
import java.lang.annotation.Annotation;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import org.apache.commons.io.IOUtils;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;

@ApplicationScoped
public class PolicyDownloadService {
    private static final int DEFAULT_INTERVAL = 30;
    public static final MediaType APPLICATION_ZIP_TYPE = new MediaType("application", "zip");
    @Inject
    private Logger log;
    @Inject
    private Event<TimerEvent> timerEvent;
    @Inject
    private AppConfiguration appConfiguration;
    @Inject
    private BaseHttpService httpService;
    @Inject
    private EncryptionService encryptionService;
    private ObjectMapper objectMapper;
    private Map<String, LoadedPolicySource> loadedPolicySourcesUris;
    private AtomicBoolean isActive;

    @PostConstruct
    public void init() {
        this.log.info("Initializing Policy Download Service ...");
        this.isActive = new AtomicBoolean(false);
        this.objectMapper = new ObjectMapper();
        this.loadedPolicySourcesUris = new HashMap<String, LoadedPolicySource>();
    }

    public void initTimer() {
        this.log.debug("Initializing Policy Download Service Timer");
        int delay = 30;
        int interval = 30;
        this.timerEvent.fire((Object)new TimerEvent(new TimerSchedule(30, 30), (Object)new PolicyDownloadEvent(), new Annotation[]{Scheduled.Literal.INSTANCE}));
    }

    public void initTime(@Observes @ApplicationInitialized(value=ApplicationScoped.class) ApplicationInitializedEvent event) {
        this.initTimer();
        this.reloadPolicies();
    }

    @Asynchronous
    public void reloadPoliciesTimerEvent(@Observes @Scheduled PolicyDownloadEvent policyDownloadEvent) {
        if (this.isActive.get()) {
            return;
        }
        if (!this.isActive.compareAndSet(false, true)) {
            return;
        }
        try {
            this.reloadPolicies();
        }
        catch (Throwable ex) {
            this.log.error("Exception happened while reloading policies", ex);
        }
        finally {
            this.isActive.set(false);
        }
    }

    private void reloadPolicies() {
        CedarlingConfiguration cedarlingConfiguration = this.appConfiguration.getCedarlingConfiguration();
        if (cedarlingConfiguration.getPolicySources() == null || cedarlingConfiguration.getPolicySources().size() == 0) {
            this.log.debug("Policies sources is not specified");
            this.cleanAllPolicies();
            return;
        }
        ArrayList<String> downloadedPolicies = new ArrayList<String>();
        for (PolicySource policySource : cedarlingConfiguration.getPolicySources()) {
            List<LoadedPolicySource> loadedPolicySources = this.loadPolicy(policySource);
            if (loadedPolicySources == null) continue;
            for (LoadedPolicySource loadedPolicySource : loadedPolicySources) {
                String policyStoreUri = loadedPolicySource.getPolicyStoreUri();
                downloadedPolicies.add(policyStoreUri);
                this.loadedPolicySourcesUris.put(policyStoreUri, loadedPolicySource);
            }
        }
        Iterator<Map.Entry<String, LoadedPolicySource>> it = this.loadedPolicySourcesUris.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry<String, LoadedPolicySource> item = it.next();
            if (downloadedPolicies.contains(item.getKey())) continue;
            it.remove();
        }
        this.log.debug("End URIs policies reload. Count policies: {}", (Object)this.loadedPolicySourcesUris.size());
    }

    private List<LoadedPolicySource> loadPolicy(PolicySource policySource) {
        String policyStoreUri = policySource.getPolicyStoreUri();
        if (!policySource.isEnabled()) {
            this.log.debug("Found disabled policy URI: {}", (Object)policyStoreUri);
            return null;
        }
        this.log.debug("Starting policy store load from URI: {}", (Object)policyStoreUri);
        if (StringHelper.isEmpty((String)policyStoreUri)) {
            this.log.warn("Policy store URI is not specified");
            return null;
        }
        String decryptedAuthorizationToken = this.encryptionService.decrypt(policySource.getAuthorizationToken(), true);
        HttpGet request = new HttpGet(policyStoreUri);
        if (StringHelper.isNotEmpty((String)decryptedAuthorizationToken)) {
            request.setHeader("Authorization", "Bearer " + decryptedAuthorizationToken);
        }
        CloseableHttpClient httpClient = this.httpService.getHttpsClient();
        try {
            CloseableHttpResponse httpResponse = httpClient.execute((HttpUriRequest)request);
            boolean result = this.httpService.isResponseStastusCodeOk((HttpResponse)httpResponse);
            if (!result) {
                this.log.error("Get invalid response from URI {}", (Object)policyStoreUri);
                return null;
            }
            MediaType responseMediaType = this.parseMediaType((HttpResponse)httpResponse);
            if (responseMediaType.isCompatible(MediaType.APPLICATION_JSON_TYPE) || responseMediaType.isCompatible(MediaType.TEXT_PLAIN_TYPE)) {
                return this.loadJsonPolicyFromResponse((HttpResponse)httpResponse, policyStoreUri);
            }
            if (responseMediaType.isCompatible(APPLICATION_ZIP_TYPE)) {
                return this.loadZipPoliciesFromResponse((HttpResponse)httpResponse, policyStoreUri);
            }
            this.log.error("Unsupported response media type from URI {}", (Object)policyStoreUri);
        }
        catch (IOException ex) {
            this.log.error("Failed to execute load policies list from URI {}", (Object)policyStoreUri, (Object)ex);
        }
        return null;
    }

    private List<LoadedPolicySource> loadJsonPolicyFromResponse(HttpResponse httpResponse, String policyStoreUri) throws IOException {
        String policyJson = EntityUtils.toString((HttpEntity)httpResponse.getEntity(), (Charset)StandardCharsets.UTF_8);
        JsonNode policyStoreJson = this.parsePolicyJson(policyStoreUri, policyJson);
        boolean valid = policyStoreJson != null;
        this.log.debug("End JSON policies load from: '{}'", (Object)policyStoreUri);
        if (valid) {
            return Arrays.asList(new LoadedPolicySource(policyStoreUri, policyStoreJson.toString()));
        }
        return null;
    }

    private List<LoadedPolicySource> loadZipPoliciesFromResponse(HttpResponse httpResponse, String policyStoreUri) throws IOException {
        ArrayList<LoadedPolicySource> result = new ArrayList<LoadedPolicySource>();
        try (ZipInputStream zis = new ZipInputStream(httpResponse.getEntity().getContent());){
            ZipEntry ze;
            while ((ze = zis.getNextEntry()) != null) {
                String fileName = ze.getName();
                if (ze.isDirectory() || !fileName.endsWith(".json") && !fileName.endsWith(".cedar")) continue;
                byte[] fileBytes = IOUtils.toByteArray((InputStream)zis);
                String zipFilePolicy = new String(fileBytes, StandardCharsets.UTF_8);
                zis.closeEntry();
                String filePath = policyStoreUri + ":" + fileName;
                boolean valid = this.parsePolicyJson(filePath, zipFilePolicy) != null;
                if (!valid) continue;
                result.add(new LoadedPolicySource(filePath, zipFilePolicy));
            }
        }
        this.log.debug("End Zip policies load from: '{}'", (Object)policyStoreUri);
        return result;
    }

    private JsonNode parsePolicyJson(String policyStoreUri, String policyJson) {
        try {
            if (policyJson.contains("\"cedar_version\"")) {
                JsonNode policyStoreJson = this.objectMapper.readTree(policyJson);
                if (!policyStoreJson.hasNonNull("cedar_version") || !policyStoreJson.hasNonNull("policy_stores")) {
                    this.log.error(String.format("Policy store is invalid: '{}'", policyStoreUri));
                    return null;
                }
                return policyStoreJson;
            }
            if (policyJson.contains("\"content\"") && policyJson.contains("\"encoding\"") && policyJson.contains("\"html\"")) {
                JsonNode responseJson = this.objectMapper.readTree(policyJson);
                if (!(responseJson.hasNonNull("content") && responseJson.hasNonNull("encoding") && "base64".equals(responseJson.get("encoding").asText()))) {
                    this.log.error(String.format("Response doesn't contains required 'content' and 'encoding' attributes : '{}'", responseJson));
                    return null;
                }
                JsonNode policyStoreJson = this.objectMapper.readTree(this.httpService.decodeBase64(responseJson.get("content").asText()));
                if (!policyStoreJson.hasNonNull("cedar_version") || !policyStoreJson.hasNonNull("policy_stores")) {
                    this.log.error(String.format("Policy store is invalid: '{}'", policyStoreUri));
                    return null;
                }
                return policyStoreJson;
            }
        }
        catch (Exception ex) {
            this.log.error(String.format("Failed to parse JSON response from URI: '{}'", policyStoreUri));
            this.log.trace(String.format("Failed to parse JSON file: '{}'", policyJson));
            return null;
        }
        this.log.error(String.format("Fetched policy doesn't contains \"cedar_version\": '{}'", policyJson));
        return null;
    }

    private MediaType parseMediaType(HttpResponse response) {
        Header contentTypeHeader = response.getFirstHeader("Content-Type");
        if (contentTypeHeader != null) {
            String contentTypeStr = contentTypeHeader.getValue();
            try {
                MediaType contentType = MediaType.valueOf((String)contentTypeStr);
                this.log.debug("Parsed content type '{}' to media type '{}'", (Object)contentTypeStr, (Object)contentType);
                return contentType;
            }
            catch (IllegalArgumentException ex) {
                this.log.error("Failed to parse content type: '{}'", (Object)contentTypeStr);
            }
        }
        return null;
    }

    private void cleanAllPolicies() {
        this.loadedPolicySourcesUris.clear();
    }

    public Map<String, LoadedPolicySource> getLoadedPolicies() {
        if (this.loadedPolicySourcesUris == null) {
            return new HashMap<String, LoadedPolicySource>();
        }
        return new HashMap<String, LoadedPolicySource>(this.loadedPolicySourcesUris);
    }

    public class LoadedPolicySource {
        private final String policyStoreUri;
        private final String policyJson;

        public LoadedPolicySource(String policyStoreUri, String policyJson) {
            this.policyStoreUri = policyStoreUri;
            this.policyJson = policyJson;
        }

        public String getPolicyJson() {
            return this.policyJson;
        }

        public String getPolicyStoreUri() {
            return this.policyStoreUri;
        }

        public String toString() {
            return "LoadedPolicySource [policyJson=" + this.policyJson + ", policyStoreUri=" + this.policyStoreUri + "]";
        }
    }
}

