/*
 * Decompiled with CFR 0.152.
 */
package com.couchbase.client.core.diagnostics;

import com.couchbase.client.core.Core;
import com.couchbase.client.core.Reactor;
import com.couchbase.client.core.annotation.Stability;
import com.couchbase.client.core.config.BucketConfig;
import com.couchbase.client.core.config.ClusterConfig;
import com.couchbase.client.core.config.GlobalConfig;
import com.couchbase.client.core.config.NodeInfo;
import com.couchbase.client.core.config.PortInfo;
import com.couchbase.client.core.diagnostics.EndpointPingReport;
import com.couchbase.client.core.diagnostics.PingResult;
import com.couchbase.client.core.diagnostics.PingState;
import com.couchbase.client.core.diagnostics.WaitUntilReadyHelper;
import com.couchbase.client.core.endpoint.http.CoreCommonOptions;
import com.couchbase.client.core.endpoint.http.CoreHttpPath;
import com.couchbase.client.core.endpoint.http.CoreHttpRequest;
import com.couchbase.client.core.error.TimeoutException;
import com.couchbase.client.core.io.CollectionIdentifier;
import com.couchbase.client.core.logging.RedactableArgument;
import com.couchbase.client.core.msg.RequestContext;
import com.couchbase.client.core.msg.RequestTarget;
import com.couchbase.client.core.msg.kv.KvPingRequest;
import com.couchbase.client.core.msg.kv.KvPingResponse;
import com.couchbase.client.core.node.NodeIdentifier;
import com.couchbase.client.core.protostellar.CoreProtostellarUtil;
import com.couchbase.client.core.retry.RetryStrategy;
import com.couchbase.client.core.service.ServiceType;
import com.couchbase.client.core.util.CbCollections;
import com.couchbase.client.core.util.CbThrowables;
import com.couchbase.client.core.util.HostAndPort;
import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.util.annotation.Nullable;

@Stability.Internal
public class HealthPinger {
    private static final Set<ServiceType> pingableServices = Collections.unmodifiableSet(EnumSet.of(ServiceType.QUERY, ServiceType.KV, ServiceType.VIEWS, ServiceType.SEARCH, ServiceType.ANALYTICS));

    @Stability.Internal
    public static Mono<PingResult> ping(Core core, Optional<Duration> timeout, RetryStrategy retryStrategy, Set<ServiceType> serviceTypes, Optional<String> reportId, Optional<String> bucketName) {
        return HealthPinger.ping(core, timeout, retryStrategy, serviceTypes, reportId, bucketName, WaitUntilReadyHelper.WaitUntilReadyLogger.dummy);
    }

    @Stability.Internal
    public static Mono<PingResult> ping(Core core, Optional<Duration> timeout, RetryStrategy retryStrategy, Set<ServiceType> serviceTypes, Optional<String> reportId, Optional<String> bucketName, WaitUntilReadyHelper.WaitUntilReadyLogger log) {
        return Mono.defer(() -> {
            if (core.isProtostellar()) {
                return Mono.error((Throwable)CoreProtostellarUtil.unsupportedCurrentlyInProtostellar());
            }
            Set<RequestTarget> targets = HealthPinger.extractPingTargets(core.clusterConfig(), serviceTypes, bucketName, log);
            return HealthPinger.pingTargets(core, targets, timeout, retryStrategy, log).collectList().map(reports -> new PingResult(reports.stream().collect(Collectors.groupingBy(EndpointPingReport::type)), core.context().environment().userAgent().formattedShort(), reportId.orElse(UUID.randomUUID().toString())));
        });
    }

    @Stability.Internal
    static Set<RequestTarget> extractPingTargets(ClusterConfig clusterConfig, @Nullable Set<ServiceType> serviceTypesOrEmpty, Optional<String> bucketName, WaitUntilReadyHelper.WaitUntilReadyLogger log) {
        EnumSet<ServiceType> serviceTypes = CbCollections.isNullOrEmpty(serviceTypesOrEmpty) ? EnumSet.allOf(ServiceType.class) : EnumSet.copyOf(serviceTypesOrEmpty);
        serviceTypes.retainAll(pingableServices);
        HashSet<RequestTarget> targets = new HashSet();
        log.message("extractPingTargets: starting ping target extraction with candidate services: " + serviceTypes);
        if (!bucketName.isPresent()) {
            if (clusterConfig.globalConfig() != null) {
                GlobalConfig globalConfig = clusterConfig.globalConfig();
                log.message("extractPingTargets: getting ping targets from global config portInfos: " + globalConfig.portInfos());
                for (PortInfo portInfo : globalConfig.portInfos()) {
                    for (Object serviceType : portInfo.ports().keySet()) {
                        if (serviceType == ServiceType.KV || serviceType == ServiceType.VIEWS) continue;
                        RequestTarget target = new RequestTarget((ServiceType)((Object)serviceType), portInfo.identifier(), null);
                        log.message("extractPingTargets: adding target from global config: " + target);
                        targets.add(target);
                    }
                }
                log.message("extractPingTargets: ping targets after scanning global config: " + targets);
            } else {
                log.message("extractPingTargets: globalConfig is absent");
            }
            for (Map.Entry entry : clusterConfig.bucketConfigs().entrySet()) {
                log.message("extractPingTargets: getting targets from bucket config via global config for bucket " + (String)entry.getKey() + " : " + entry.getValue());
                for (NodeInfo nodeInfo : ((BucketConfig)entry.getValue()).nodes()) {
                    for (ServiceType serviceType : nodeInfo.services().keySet()) {
                        if (serviceType == ServiceType.KV || serviceType == ServiceType.VIEWS) continue;
                        RequestTarget target = new RequestTarget(serviceType, nodeInfo.identifier(), null);
                        log.message("extractPingTargets: adding target from bucket config via global config: " + target);
                        targets.add(new RequestTarget(serviceType, nodeInfo.identifier(), null));
                    }
                }
            }
        } else {
            BucketConfig bucketConfig = clusterConfig.bucketConfig(bucketName.get());
            if (bucketConfig != null) {
                log.message("extractPingTargets: Getting targets from bucket config: " + bucketConfig);
                for (NodeInfo nodeInfo : bucketConfig.nodes()) {
                    for (ServiceType serviceType : nodeInfo.services().keySet()) {
                        RequestTarget target = serviceType != ServiceType.VIEWS && serviceType != ServiceType.KV ? new RequestTarget(serviceType, nodeInfo.identifier(), null) : new RequestTarget(serviceType, nodeInfo.identifier(), bucketName.get());
                        log.message("extractPingTargets: adding target from bucket config: " + target);
                        targets.add(target);
                    }
                }
            } else {
                log.message("extractPingTargets: Bucket name was present, but clusterConfig has no config for bucket " + bucketName);
            }
        }
        targets = targets.stream().filter(t -> serviceTypes.contains((Object)t.serviceType())).collect(Collectors.toSet());
        log.message("extractPingTargets: Finished. Returning filtered targets (grouped by node): " + HealthPinger.formatGroupedByNode(targets));
        return targets;
    }

    private static String format(NodeIdentifier id) {
        return new HostAndPort(id.address(), id.managerPort()).format();
    }

    static Map<String, List<String>> formatGroupedByNode(Collection<RequestTarget> targets) {
        Map<String, List<RequestTarget>> grouped = targets.stream().collect(Collectors.groupingBy(requestTarget -> RedactableArgument.redactSystem(HealthPinger.format(requestTarget.nodeIdentifier())).toString()));
        return CbCollections.transformValues(grouped, it -> CbCollections.transform(it, target -> target.serviceType().toString()));
    }

    private static Flux<EndpointPingReport> pingTargets(Core core, Set<RequestTarget> targets, Optional<Duration> timeout, RetryStrategy retryStrategy, WaitUntilReadyHelper.WaitUntilReadyLogger log) {
        CoreCommonOptions options = CoreCommonOptions.of(timeout.orElse(null), retryStrategy, null);
        return Flux.fromIterable(targets).flatMap(target -> HealthPinger.pingTarget(core, target, options, log));
    }

    static Mono<EndpointPingReport> pingTarget(Core core, RequestTarget target, CoreCommonOptions options, WaitUntilReadyHelper.WaitUntilReadyLogger log) {
        switch (target.serviceType()) {
            case QUERY: 
            case ANALYTICS: {
                return HealthPinger.pingHttpEndpoint(core, target, options, "/admin/ping", log);
            }
            case KV: {
                return HealthPinger.pingKv(core, target, options, log);
            }
            case VIEWS: {
                return HealthPinger.pingHttpEndpoint(core, target, options, "/", log);
            }
            case SEARCH: {
                return HealthPinger.pingHttpEndpoint(core, target, options, "/api/ping", log);
            }
        }
        return Mono.error((Throwable)new RuntimeException("Don't know how to ping the " + (Object)((Object)target.serviceType()) + " service."));
    }

    private static EndpointPingReport assembleSuccessReport(RequestContext context, String channelId, Optional<String> namespace) {
        String dispatchTo = null;
        String dispatchFrom = null;
        if (context.lastDispatchedTo() != null) {
            dispatchTo = context.lastDispatchedTo().toString();
        }
        if (context.lastDispatchedFrom() != null) {
            dispatchFrom = context.lastDispatchedFrom().toString();
        }
        return new EndpointPingReport(context.request().serviceType(), "0x" + channelId, dispatchFrom, dispatchTo, PingState.OK, namespace, Duration.ofNanos(context.logicalRequestLatency()), Optional.empty());
    }

    private static EndpointPingReport assembleFailureReport(Throwable throwable, RequestContext context, Optional<String> namespace) {
        String dispatchTo = null;
        String dispatchFrom = null;
        if (context.lastDispatchedTo() != null) {
            dispatchTo = context.lastDispatchedTo().toString();
        }
        if (context.lastDispatchedFrom() != null) {
            dispatchFrom = context.lastDispatchedFrom().toString();
        }
        PingState state = throwable instanceof TimeoutException ? PingState.TIMEOUT : PingState.ERROR;
        return new EndpointPingReport(context.request().serviceType(), null, dispatchFrom, dispatchTo, state, namespace, state == PingState.TIMEOUT ? context.request().timeout() : Duration.ofNanos(context.logicalRequestLatency()), Optional.empty());
    }

    private static Mono<EndpointPingReport> pingKv(Core core, RequestTarget target, CoreCommonOptions options, WaitUntilReadyHelper.WaitUntilReadyLogger log) {
        return Mono.defer(() -> {
            Duration timeout = options.timeout().orElse(core.context().environment().timeoutConfig().kvTimeout());
            CollectionIdentifier collectionIdentifier = CollectionIdentifier.fromDefault(target.bucketName());
            KvPingRequest request = new KvPingRequest(timeout, core.context(), (RetryStrategy)options.retryStrategy().orElse(null), collectionIdentifier, target.nodeIdentifier());
            core.send(request);
            return Reactor.wrap(request, request.response(), true).map(response -> {
                request.context().logicallyComplete();
                EndpointPingReport report = HealthPinger.assembleSuccessReport(request.context(), ((KvPingResponse)response).channelId(), Optional.ofNullable(target.bucketName()));
                log.message("ping: Ping succeeded for " + target + " ; " + report);
                return report;
            }).onErrorResume(throwable -> {
                request.context().logicallyComplete((Throwable)throwable);
                EndpointPingReport report = HealthPinger.assembleFailureReport(throwable, request.context(), Optional.ofNullable(target.bucketName()));
                log.message("ping: Ping failed for " + target + " ; " + report + " ; " + CbThrowables.getStackTraceAsString(throwable));
                return Mono.just((Object)report);
            });
        });
    }

    private static Mono<EndpointPingReport> pingHttpEndpoint(Core core, RequestTarget target, CoreCommonOptions options, String path, WaitUntilReadyHelper.WaitUntilReadyLogger log) {
        return Mono.defer(() -> {
            log.message("ping: Pinging " + target);
            CoreHttpRequest request = core.httpClient(target).get(CoreHttpPath.path(path), options).build();
            core.send(request);
            return Reactor.wrap(request, request.response(), true).map(response -> {
                request.context().logicallyComplete();
                EndpointPingReport report = HealthPinger.assembleSuccessReport(request.context(), response.channelId(), Optional.empty());
                log.message("ping: Ping succeeded for " + target + " ; " + report);
                return report;
            }).onErrorResume(throwable -> {
                request.context().logicallyComplete((Throwable)throwable);
                EndpointPingReport report = HealthPinger.assembleFailureReport(throwable, request.context(), Optional.empty());
                log.message("ping: Ping failed for " + target + " ; " + report + " ; " + CbThrowables.getStackTraceAsString(throwable));
                return Mono.just((Object)report);
            });
        });
    }
}

