/*
 * Decompiled with CFR 0.152.
 */
package io.jans.orm.operation.auth;

import io.jans.orm.operation.auth.Argon2EncodingUtils;
import io.jans.orm.operation.auth.BCrypt;
import io.jans.orm.operation.auth.PasswordDetails;
import io.jans.orm.operation.auth.PasswordEncryptionMethod;
import io.jans.orm.util.StringHelper;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Base64;
import java.util.HashMap;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.PBEKeySpec;
import org.apache.commons.codec.digest.Crypt;
import org.bouncycastle.crypto.generators.Argon2BytesGenerator;
import org.bouncycastle.crypto.params.Argon2Parameters;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class PasswordEncryptionHelper {
    private static final Logger LOG = LoggerFactory.getLogger(PasswordEncryptionHelper.class);
    private static final byte[] CRYPT_SALT_CHARS = StringHelper.getBytesUtf8((String)"./0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz");
    private static final HashMap<String, String> overrideParameters = new HashMap();
    public static final String ARGON2_TYPE_KEY = "argon2.type";
    public static final String ARGON2_VERSION_KEY = "argon2.version";
    public static final String ARGON2_SALT_LENGTH_KEY = "argon2.salt-length";
    public static final String ARGON2_MEMORY_KEY = "argon2.memory";
    public static final String ARGON2_ITERATIONS_KEY = "argon2.iterations";
    public static final String ARGON2_PARALLELISM_KEY = "argon2.parallelism";
    public static final String ARGON2_HASH_LENGTH_KEY = "argon2.hash-length";

    private PasswordEncryptionHelper() {
    }

    public static void configureParameters(HashMap<String, String> customOverrideParameters) {
        overrideParameters.putAll(customOverrideParameters);
    }

    public static PasswordEncryptionMethod findAlgorithm(String credentials) {
        return PasswordEncryptionHelper.findAlgorithm(StringHelper.getBytesUtf8((String)credentials));
    }

    public static PasswordEncryptionMethod findAlgorithm(byte[] credentials) {
        String algorithm = PasswordEncryptionHelper.findAlgorithmString(credentials);
        if (algorithm != null) {
            return PasswordEncryptionMethod.getMethod(algorithm);
        }
        return null;
    }

    public static String findAlgorithmString(String credentials) {
        return PasswordEncryptionHelper.findAlgorithmString(StringHelper.getBytesUtf8((String)credentials));
    }

    public static String findAlgorithmString(byte[] credentials) {
        if (credentials == null || credentials.length == 0) {
            return null;
        }
        if (credentials[0] == 123) {
            int pos;
            for (pos = 1; pos < credentials.length && credentials[pos] != 125; ++pos) {
            }
            if (pos < credentials.length) {
                if (pos == 1) {
                    return null;
                }
                Object algorithm = StringHelper.toLowerCase((String)StringHelper.utf8ToString((byte[])credentials, (int)1, (int)(pos - 1)));
                if (credentials.length > pos + 3 && credentials[pos + 1] == 36 && Character.isDigit(credentials[pos + 2])) {
                    if (credentials[pos + 3] == 36) {
                        algorithm = (String)algorithm + StringHelper.utf8ToString((byte[])credentials, (int)(pos + 1), (int)3);
                    } else if (credentials.length > pos + 4 && credentials[pos + 4] == 36) {
                        algorithm = (String)algorithm + StringHelper.utf8ToString((byte[])credentials, (int)(pos + 1), (int)4);
                    }
                }
                return algorithm;
            }
            return null;
        }
        return null;
    }

    public static String createStoragePassword(String credentials, PasswordEncryptionMethod algorithm) {
        byte[] resultBytes = PasswordEncryptionHelper.createStoragePassword(StringHelper.getBytesUtf8((String)credentials), algorithm);
        return StringHelper.utf8ToString((byte[])resultBytes);
    }

    public static byte[] createStoragePassword(byte[] credentials, PasswordEncryptionMethod algorithm) {
        byte[] salt;
        if (algorithm == null) {
            return credentials;
        }
        Argon2Parameters extendedParameter = null;
        switch (algorithm) {
            case HASH_METHOD_SSHA: 
            case HASH_METHOD_SSHA256: 
            case HASH_METHOD_SSHA384: 
            case HASH_METHOD_SSHA512: 
            case HASH_METHOD_SMD5: {
                salt = new byte[8];
                new SecureRandom().nextBytes(salt);
                break;
            }
            case HASH_METHOD_PKCS5S2: {
                salt = new byte[16];
                new SecureRandom().nextBytes(salt);
                break;
            }
            case HASH_METHOD_CRYPT: {
                salt = PasswordEncryptionHelper.generateCryptSalt(2);
                break;
            }
            case HASH_METHOD_CRYPT_MD5: 
            case HASH_METHOD_CRYPT_SHA256: 
            case HASH_METHOD_CRYPT_SHA512: {
                salt = PasswordEncryptionHelper.generateCryptSalt(8);
                break;
            }
            case HASH_METHOD_CRYPT_BCRYPT: 
            case HASH_METHOD_CRYPT_BCRYPT_B: {
                salt = StringHelper.getBytesUtf8((String)BCrypt.genSalt());
                break;
            }
            case HASH_METHOD_ARGON2: {
                int type = PasswordEncryptionHelper.getParameterInteger(ARGON2_TYPE_KEY, 2);
                int version = PasswordEncryptionHelper.getParameterInteger(ARGON2_VERSION_KEY, 19);
                int saltLength = PasswordEncryptionHelper.getParameterInteger(ARGON2_SALT_LENGTH_KEY, 16);
                int memory = PasswordEncryptionHelper.getParameterInteger(ARGON2_MEMORY_KEY, 7168);
                int iterations = PasswordEncryptionHelper.getParameterInteger(ARGON2_ITERATIONS_KEY, 5);
                int parallelism = PasswordEncryptionHelper.getParameterInteger(ARGON2_PARALLELISM_KEY, 1);
                salt = new byte[saltLength];
                new SecureRandom().nextBytes(salt);
                Argon2Parameters.Builder paramsBuilder = new Argon2Parameters.Builder(type).withVersion(version).withIterations(iterations).withMemoryAsKB(memory).withParallelism(parallelism).withSalt(salt);
                extendedParameter = paramsBuilder.build();
                break;
            }
            default: {
                salt = null;
            }
        }
        byte[] hashedPassword = PasswordEncryptionHelper.encryptPassword(credentials, algorithm, extendedParameter, salt);
        StringBuilder sb = new StringBuilder();
        sb.append('{').append(StringHelper.toUpperCase((String)algorithm.getPrefix())).append('}');
        if (algorithm == PasswordEncryptionMethod.HASH_METHOD_CRYPT || algorithm == PasswordEncryptionMethod.HASH_METHOD_CRYPT_BCRYPT) {
            sb.append(StringHelper.utf8ToString((byte[])salt));
            sb.append(StringHelper.utf8ToString((byte[])hashedPassword));
        } else if (algorithm == PasswordEncryptionMethod.HASH_METHOD_CRYPT_MD5 || algorithm == PasswordEncryptionMethod.HASH_METHOD_CRYPT_SHA256 || algorithm == PasswordEncryptionMethod.HASH_METHOD_CRYPT_SHA512) {
            sb.append(algorithm.getSubPrefix());
            sb.append(StringHelper.utf8ToString((byte[])salt));
            sb.append('$');
            sb.append(StringHelper.utf8ToString((byte[])hashedPassword));
        } else if (algorithm == PasswordEncryptionMethod.HASH_METHOD_ARGON2) {
            String encodedHash = Argon2EncodingUtils.encode(hashedPassword, extendedParameter);
            sb.append(Base64.getEncoder().encodeToString(StringHelper.getBytesUtf8((String)encodedHash)));
        } else if (salt != null) {
            byte[] hashedPasswordWithSaltBytes = new byte[hashedPassword.length + salt.length];
            if (algorithm == PasswordEncryptionMethod.HASH_METHOD_PKCS5S2) {
                PasswordEncryptionHelper.merge(hashedPasswordWithSaltBytes, salt, hashedPassword);
            } else {
                PasswordEncryptionHelper.merge(hashedPasswordWithSaltBytes, hashedPassword, salt);
            }
            sb.append(String.valueOf(Base64.getEncoder().encodeToString(hashedPasswordWithSaltBytes)));
        } else {
            sb.append(String.valueOf(Base64.getEncoder().encodeToString(hashedPassword)));
        }
        return StringHelper.getBytesUtf8((String)sb.toString());
    }

    public static boolean compareCredentials(String receivedCredentials, String storedCredentials) {
        return PasswordEncryptionHelper.compareCredentials(StringHelper.getBytesUtf8((String)receivedCredentials), StringHelper.getBytesUtf8((String)storedCredentials));
    }

    public static boolean compareCredentials(byte[] receivedCredentials, byte[] storedCredentials) {
        PasswordEncryptionMethod algorithm = PasswordEncryptionHelper.findAlgorithm(storedCredentials);
        if (algorithm != null) {
            PasswordDetails passwordDetails = PasswordEncryptionHelper.splitCredentials(storedCredentials);
            byte[] userPassword = PasswordEncryptionHelper.encryptPassword(receivedCredentials, passwordDetails.getAlgorithm(), passwordDetails.getExtendedParameters(), passwordDetails.getSalt());
            return PasswordEncryptionHelper.compareBytes(userPassword, passwordDetails.getPassword());
        }
        return PasswordEncryptionHelper.compareBytes(receivedCredentials, storedCredentials);
    }

    private static boolean compareBytes(byte[] provided, byte[] stored) {
        if (stored == null) {
            return provided == null;
        }
        if (provided == null) {
            return false;
        }
        if (stored.length != provided.length) {
            return false;
        }
        int result = 0;
        for (int i = 0; i < stored.length; ++i) {
            result |= stored[i] ^ provided[i];
        }
        return result == 0;
    }

    private static byte[] encryptPassword(byte[] credentials, PasswordEncryptionMethod algorithm, Object extendedParameters, byte[] salt) {
        switch (algorithm) {
            case HASH_METHOD_SSHA: 
            case HASH_METHOD_SHA: {
                return PasswordEncryptionHelper.digest(PasswordEncryptionMethod.HASH_METHOD_SHA, credentials, salt);
            }
            case HASH_METHOD_SSHA256: 
            case HASH_METHOD_SHA256: {
                return PasswordEncryptionHelper.digest(PasswordEncryptionMethod.HASH_METHOD_SHA256, credentials, salt);
            }
            case HASH_METHOD_SSHA384: 
            case HASH_METHOD_SHA384: {
                return PasswordEncryptionHelper.digest(PasswordEncryptionMethod.HASH_METHOD_SHA384, credentials, salt);
            }
            case HASH_METHOD_SSHA512: 
            case HASH_METHOD_SHA512: {
                return PasswordEncryptionHelper.digest(PasswordEncryptionMethod.HASH_METHOD_SHA512, credentials, salt);
            }
            case HASH_METHOD_SMD5: 
            case HASH_METHOD_MD5: {
                return PasswordEncryptionHelper.digest(PasswordEncryptionMethod.HASH_METHOD_MD5, credentials, salt);
            }
            case HASH_METHOD_CRYPT: {
                String saltWithCrypted = Crypt.crypt((String)StringHelper.utf8ToString((byte[])credentials), (String)StringHelper.utf8ToString((byte[])salt));
                String crypted = saltWithCrypted.substring(2);
                return StringHelper.getBytesUtf8((String)crypted);
            }
            case HASH_METHOD_CRYPT_MD5: 
            case HASH_METHOD_CRYPT_SHA256: 
            case HASH_METHOD_CRYPT_SHA512: {
                String saltWithCrypted2 = Crypt.crypt((String)StringHelper.utf8ToString((byte[])credentials), (String)(algorithm.getSubPrefix() + StringHelper.utf8ToString((byte[])salt)));
                String crypted2 = saltWithCrypted2.substring(saltWithCrypted2.lastIndexOf(36) + 1);
                return StringHelper.getBytesUtf8((String)crypted2);
            }
            case HASH_METHOD_CRYPT_BCRYPT: 
            case HASH_METHOD_CRYPT_BCRYPT_B: {
                String crypted3 = BCrypt.hashPw(StringHelper.utf8ToString((byte[])credentials), StringHelper.utf8ToString((byte[])salt));
                return StringHelper.getBytesUtf8((String)crypted3.substring(crypted3.length() - 31));
            }
            case HASH_METHOD_PKCS5S2: {
                return PasswordEncryptionHelper.generatePbkdf2Hash(credentials, algorithm, salt);
            }
            case HASH_METHOD_ARGON2: {
                if (extendedParameters instanceof Argon2Parameters) {
                    Argon2Parameters argon2Parameters;
                    Argon2Parameters parameters = argon2Parameters = (Argon2Parameters)extendedParameters;
                    int hashLength = 32;
                    hashLength = argon2Parameters.getSecret() != null ? argon2Parameters.getSecret().length : PasswordEncryptionHelper.getParameterInteger(ARGON2_HASH_LENGTH_KEY, hashLength);
                    byte[] argon2Hash = PasswordEncryptionHelper.generateArgon2Hash(credentials, parameters, hashLength);
                    return argon2Hash;
                }
                throw new IllegalArgumentException("Invalid Argon2Parameters specified");
            }
        }
        return credentials;
    }

    private static byte[] digest(PasswordEncryptionMethod algorithm, byte[] password, byte[] salt) {
        MessageDigest digest;
        try {
            digest = MessageDigest.getInstance(algorithm.getAlgorithm());
        }
        catch (NoSuchAlgorithmException ex) {
            return null;
        }
        if (salt != null) {
            digest.update(password);
            digest.update(salt);
            return digest.digest();
        }
        return digest.digest(password);
    }

    public static PasswordDetails splitCredentials(byte[] credentials) {
        PasswordEncryptionMethod algorithm = PasswordEncryptionHelper.findAlgorithm(credentials);
        if (algorithm == null) {
            return new PasswordDetails(null, null, credentials);
        }
        int algoLength = algorithm.getPrefix().length() + 2;
        switch (algorithm) {
            case HASH_METHOD_SMD5: 
            case HASH_METHOD_MD5: {
                return PasswordEncryptionHelper.getCredentials(credentials, algoLength, algorithm.getHashLength(), algorithm);
            }
            case HASH_METHOD_SSHA: 
            case HASH_METHOD_SHA: {
                return PasswordEncryptionHelper.getCredentials(credentials, algoLength, algorithm.getHashLength(), algorithm);
            }
            case HASH_METHOD_SSHA256: 
            case HASH_METHOD_SHA256: {
                return PasswordEncryptionHelper.getCredentials(credentials, algoLength, algorithm.getHashLength(), algorithm);
            }
            case HASH_METHOD_SSHA384: 
            case HASH_METHOD_SHA384: {
                return PasswordEncryptionHelper.getCredentials(credentials, algoLength, algorithm.getHashLength(), algorithm);
            }
            case HASH_METHOD_SSHA512: 
            case HASH_METHOD_SHA512: {
                return PasswordEncryptionHelper.getCredentials(credentials, algoLength, algorithm.getHashLength(), algorithm);
            }
            case HASH_METHOD_PKCS5S2: {
                return PasswordEncryptionHelper.getPbkdf2Credentials(credentials, algoLength, algorithm);
            }
            case HASH_METHOD_CRYPT: {
                byte[] salt = new byte[2];
                byte[] password = new byte[credentials.length - salt.length - algoLength];
                PasswordEncryptionHelper.split(credentials, algoLength, salt, password);
                return new PasswordDetails(algorithm, salt, password);
            }
            case HASH_METHOD_CRYPT_BCRYPT: 
            case HASH_METHOD_CRYPT_BCRYPT_B: {
                byte[] salt = Arrays.copyOfRange(credentials, algoLength, credentials.length - 31);
                byte[] password = Arrays.copyOfRange(credentials, credentials.length - 31, credentials.length);
                return new PasswordDetails(algorithm, salt, password);
            }
            case HASH_METHOD_CRYPT_MD5: 
            case HASH_METHOD_CRYPT_SHA256: 
            case HASH_METHOD_CRYPT_SHA512: {
                return PasswordEncryptionHelper.getCryptCredentials(credentials, algoLength += 3, algorithm);
            }
            case HASH_METHOD_ARGON2: {
                return PasswordEncryptionHelper.getArgon2Credentials(credentials, algoLength, algorithm);
            }
        }
        throw new IllegalArgumentException("Unknown hash algorithm " + algorithm);
    }

    private static PasswordDetails getCredentials(byte[] credentials, int algoLength, int hashLen, PasswordEncryptionMethod algorithm) {
        byte[] passwordAndSalt = Base64.getDecoder().decode(StringHelper.utf8ToString((byte[])credentials, (int)algoLength, (int)(credentials.length - algoLength)));
        int saltLength = passwordAndSalt.length - hashLen;
        byte[] salt = saltLength == 0 ? null : new byte[saltLength];
        byte[] password = new byte[hashLen];
        PasswordEncryptionHelper.split(passwordAndSalt, 0, password, salt);
        return new PasswordDetails(algorithm, salt, password);
    }

    private static void split(byte[] all, int offset, byte[] left, byte[] right) {
        System.arraycopy(all, offset, left, 0, left.length);
        if (right != null) {
            System.arraycopy(all, offset + left.length, right, 0, right.length);
        }
    }

    private static void merge(byte[] all, byte[] left, byte[] right) {
        System.arraycopy(left, 0, all, 0, left.length);
        System.arraycopy(right, 0, all, left.length, right.length);
    }

    private static byte[] generatePbkdf2Hash(byte[] credentials, PasswordEncryptionMethod algorithm, byte[] salt) {
        try {
            SecretKeyFactory sk = SecretKeyFactory.getInstance(algorithm.getAlgorithm());
            char[] password = StringHelper.utf8ToString((byte[])credentials).toCharArray();
            PBEKeySpec keySpec = new PBEKeySpec(password, salt, 10000, algorithm.getHashLength() * 8);
            SecretKey key = sk.generateSecret(keySpec);
            return key.getEncoded();
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private static byte[] generateArgon2Hash(byte[] credentials, Argon2Parameters parameters, int hashLength) {
        Argon2BytesGenerator generator = new Argon2BytesGenerator();
        generator.init(parameters);
        byte[] result = new byte[hashLength];
        generator.generateBytes(credentials, result);
        return result;
    }

    private static PasswordDetails getArgon2Credentials(byte[] credentials, int algoLength, PasswordEncryptionMethod algorithm) {
        byte[] encodedHash = Base64.getDecoder().decode(StringHelper.utf8ToString((byte[])credentials, (int)algoLength, (int)(credentials.length - algoLength)));
        return Argon2EncodingUtils.decode(algorithm, StringHelper.utf8ToString((byte[])encodedHash));
    }

    private static PasswordDetails getPbkdf2Credentials(byte[] credentials, int algoLength, PasswordEncryptionMethod algorithm) {
        byte[] passwordAndSalt = Base64.getDecoder().decode(StringHelper.utf8ToString((byte[])credentials, (int)algoLength, (int)(credentials.length - algoLength)));
        int saltLength = passwordAndSalt.length - algorithm.getHashLength();
        byte[] salt = new byte[saltLength];
        byte[] password = new byte[algorithm.getHashLength()];
        PasswordEncryptionHelper.split(passwordAndSalt, 0, salt, password);
        return new PasswordDetails(algorithm, salt, password);
    }

    private static byte[] generateCryptSalt(int length) {
        byte[] salt = new byte[length];
        SecureRandom sr = new SecureRandom();
        for (int i = 0; i < salt.length; ++i) {
            salt[i] = CRYPT_SALT_CHARS[sr.nextInt(CRYPT_SALT_CHARS.length)];
        }
        return salt;
    }

    private static PasswordDetails getCryptCredentials(byte[] credentials, int algoLength, PasswordEncryptionMethod algorithm) {
        int pos;
        for (pos = algoLength; pos < credentials.length && credentials[pos] != 36; ++pos) {
        }
        byte[] salt = Arrays.copyOfRange(credentials, algoLength, pos);
        byte[] password = Arrays.copyOfRange(credentials, pos + 1, credentials.length);
        return new PasswordDetails(algorithm, salt, password);
    }

    private static int getParameterInteger(String name, int defaultValue) {
        if (!overrideParameters.containsKey(name)) {
            return defaultValue;
        }
        String value = overrideParameters.get(name);
        return StringHelper.toInteger((String)value, (int)defaultValue);
    }
}

