/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.sql.core;

import com.google.auth.oauth2.AccessToken;
import com.google.auth.oauth2.GoogleCredentials;
import com.google.cloud.sql.AuthType;
import com.google.cloud.sql.CredentialFactory;
import com.google.cloud.sql.core.AccessTokenSupplier;
import com.google.cloud.sql.core.RefreshCalculator;
import com.google.cloud.sql.core.RetryingCallable;
import java.io.IOException;
import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Date;
import java.util.Optional;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class DefaultAccessTokenSupplier
implements AccessTokenSupplier {
    private static final Logger logger = LoggerFactory.getLogger(DefaultAccessTokenSupplier.class);
    private static final String SQL_LOGIN_SCOPE = "https://www.googleapis.com/auth/sqlservice.login";
    private final CredentialFactory credentialFactory;

    static AccessTokenSupplier newInstance(AuthType authType, CredentialFactory tokenSourceFactory) {
        if (authType == AuthType.IAM) {
            return new DefaultAccessTokenSupplier(tokenSourceFactory);
        }
        return Optional::empty;
    }

    DefaultAccessTokenSupplier(CredentialFactory tokenSource) {
        this.credentialFactory = tokenSource;
    }

    @Override
    public Optional<AccessToken> get() throws IOException {
        if (this.credentialFactory == null) {
            return Optional.empty();
        }
        RetryingCallable<Optional> retries = new RetryingCallable<Optional>(() -> {
            GoogleCredentials credentials = this.credentialFactory.getCredentials();
            try {
                this.refreshIfRequired(credentials);
            }
            catch (IllegalStateException e) {
                throw new IllegalStateException("Error refreshing credentials " + credentials, e);
            }
            if (DefaultAccessTokenSupplier.isAccessTokenEmpty(credentials)) {
                String errorMessage = "Access Token has length of zero";
                logger.debug(errorMessage);
                throw new IllegalStateException(errorMessage);
            }
            this.validateAccessTokenExpiration(credentials.getAccessToken());
            GoogleCredentials downscoped = DefaultAccessTokenSupplier.getDownscopedCredentials(credentials);
            if (DefaultAccessTokenSupplier.isAccessTokenEmpty(downscoped)) {
                try {
                    downscoped.refresh();
                }
                catch (Exception e) {
                    throw new IllegalStateException("Error refreshing downscoped credentials " + credentials, e);
                }
                if (DefaultAccessTokenSupplier.isAccessTokenEmpty(downscoped)) {
                    String errorMessage = "Downscoped access token has length of zero";
                    logger.debug(errorMessage);
                    throw new IllegalStateException(errorMessage + ": " + downscoped.getClass().getName() + " from " + credentials.getClass().getName());
                }
                this.validateAccessTokenExpiration(downscoped.getAccessToken());
            }
            return Optional.of(downscoped.getAccessToken());
        });
        try {
            return retries.call();
        }
        catch (IOException | RuntimeException e) {
            throw e;
        }
        catch (Exception e) {
            throw new RuntimeException("Unexpected exception refreshing authentication token", e);
        }
    }

    private static boolean isAccessTokenEmpty(GoogleCredentials credentials) {
        return credentials.getAccessToken() == null || "".equals(credentials.getAccessToken().getTokenValue());
    }

    private void refreshIfRequired(GoogleCredentials credentials) throws IOException {
        if (credentials.getAccessToken() == null) {
            logger.debug("Current IAM AuthN Token is not set. Refreshing the token.");
            credentials.refresh();
        } else if (credentials.getAccessToken().getExpirationTime() != null && credentials.getAccessToken().getExpirationTime().toInstant().minus(RefreshCalculator.DEFAULT_REFRESH_BUFFER).isBefore(Instant.now())) {
            logger.debug("Current IAM AuthN Token expires in less than 4 minutes. Refreshing the token.");
            credentials.refresh();
        }
    }

    private void validateAccessTokenExpiration(AccessToken accessToken) {
        Instant now;
        Instant expirationTime;
        Date expirationTimeDate = accessToken.getExpirationTime();
        if (expirationTimeDate != null && ((expirationTime = expirationTimeDate.toInstant()).isBefore(now = Instant.now()) || expirationTime.equals(now))) {
            DateTimeFormatter formatter = DateTimeFormatter.ISO_INSTANT.withZone(ZoneId.of("UTC"));
            String nowFormat = formatter.format(now);
            String expirationFormat = formatter.format(expirationTime);
            String errorMessage = "Access Token expiration time is in the past. Now = " + nowFormat + " Expiration = " + expirationFormat;
            logger.debug(errorMessage);
            throw new IllegalStateException(errorMessage);
        }
    }

    static Optional<Instant> getTokenExpirationTime(Optional<AccessToken> token) {
        return token.flatMap(at -> Optional.ofNullable(at.getExpirationTime())).map(Date::toInstant);
    }

    static GoogleCredentials getDownscopedCredentials(GoogleCredentials credentials) {
        return credentials.createScoped(new String[]{SQL_LOGIN_SCOPE});
    }
}

