/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.sql.core;

import com.google.cloud.sql.ConnectorConfig;
import com.google.cloud.sql.CredentialFactory;
import com.google.cloud.sql.RefreshStrategy;
import com.google.cloud.sql.core.CloudSqlInstanceName;
import com.google.cloud.sql.core.ConnectionConfig;
import com.google.cloud.sql.core.ConnectionInfoCache;
import com.google.cloud.sql.core.ConnectionInfoRepository;
import com.google.cloud.sql.core.ConnectionInfoRepositoryFactory;
import com.google.cloud.sql.core.ConnectionMetadata;
import com.google.cloud.sql.core.InstanceConnectionNameResolver;
import com.google.cloud.sql.core.LazyRefreshConnectionInfoCache;
import com.google.cloud.sql.core.MonitoredCache;
import com.google.cloud.sql.core.ProtocolHandler;
import com.google.cloud.sql.core.RefreshAheadConnectionInfoCache;
import com.google.common.base.Strings;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningScheduledExecutorService;
import java.io.File;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.security.KeyPair;
import java.util.Timer;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.function.Function;
import javax.net.ssl.SSLSocket;
import jnr.unixsocket.UnixSocketAddress;
import jnr.unixsocket.UnixSocketChannel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class Connector {
    private static final Logger logger = LoggerFactory.getLogger(Connector.class);
    private final ConnectionInfoRepository adminApi;
    private final CredentialFactory instanceCredentialFactory;
    private final ListeningScheduledExecutorService executor;
    private final ListenableFuture<KeyPair> localKeyPair;
    private final long minRefreshDelayMs;
    private final ConcurrentHashMap<ConnectionConfig, MonitoredCache> instances = new ConcurrentHashMap();
    private final int serverProxyPort;
    private final ConnectorConfig config;
    private final InstanceConnectionNameResolver instanceNameResolver;
    private final Timer instanceNameResolverTimer;
    private final ProtocolHandler mdxProtocolHandler;

    Connector(ConnectorConfig config, ConnectionInfoRepositoryFactory connectionInfoRepositoryFactory, CredentialFactory instanceCredentialFactory, ListeningScheduledExecutorService executor, ListenableFuture<KeyPair> localKeyPair, long minRefreshDelayMs, long refreshTimeoutMs, int serverProxyPort, InstanceConnectionNameResolver instanceNameResolver, ProtocolHandler mdxProtocolHandler) {
        this.config = config;
        this.adminApi = connectionInfoRepositoryFactory.create(instanceCredentialFactory.create(), config);
        this.instanceCredentialFactory = instanceCredentialFactory;
        this.executor = executor;
        this.localKeyPair = localKeyPair;
        this.minRefreshDelayMs = minRefreshDelayMs;
        this.serverProxyPort = serverProxyPort;
        this.instanceNameResolver = instanceNameResolver;
        this.instanceNameResolverTimer = new Timer("InstanceNameResolverTimer", true);
        this.mdxProtocolHandler = mdxProtocolHandler;
    }

    public ConnectorConfig getConfig() {
        return this.config;
    }

    private String getUnixSocketArg(ConnectionConfig config) {
        String unixSocketPath = config.getUnixSocketPath();
        if (unixSocketPath != null) {
            return unixSocketPath;
        }
        if (System.getenv("CLOUD_SQL_FORCE_UNIX_SOCKET") != null) {
            logger.debug(String.format("\"CLOUD_SQL_FORCE_UNIX_SOCKET\" env var has been deprecated. Please use '%s=\"/cloudsql/INSTANCE_CONNECTION_NAME\"' property in your JDBC url instead.", "unixSocketPath"));
            return "/cloudsql/" + config.getCloudSqlInstance();
        }
        return null;
    }

    Socket connect(ConnectionConfig config, long timeoutMs) throws IOException {
        String unixSocket = this.getUnixSocketArg(config);
        String unixPathSuffix = config.getUnixSocketPathSuffix();
        if (unixSocket != null) {
            if (unixPathSuffix != null && !unixSocket.endsWith(unixPathSuffix)) {
                unixSocket = unixSocket + unixPathSuffix;
            }
            logger.debug(String.format("Connecting to Cloud SQL instance [%s] via unix socket at %s.", config.getCloudSqlInstance(), unixSocket));
            UnixSocketAddress socketAddress = new UnixSocketAddress(new File(unixSocket));
            return UnixSocketChannel.open((UnixSocketAddress)socketAddress).socket();
        }
        MonitoredCache instance = this.getConnection(config);
        try {
            ConnectionMetadata metadata = instance.getConnectionMetadata(timeoutMs);
            String instanceIp = metadata.getPreferredIpAddress();
            logger.debug(String.format("[%s] Connecting to instance.", instanceIp));
            SSLSocket socket = (SSLSocket)metadata.getSslContext().getSocketFactory().createSocket();
            socket.setKeepAlive(true);
            socket.setTcpNoDelay(true);
            socket.connect(new InetSocketAddress(instanceIp, this.serverProxyPort));
            try {
                socket.startHandshake();
            }
            catch (IOException e) {
                logger.debug("TLS handshake failed!");
                throw e;
            }
            if (metadata.isMdxClientProtocolTypeSupport() && !Strings.isNullOrEmpty((String)config.getMdxClientProtocolType())) {
                socket = this.mdxProtocolHandler.connect(socket, config.getMdxClientProtocolType());
            }
            logger.debug(String.format("[%s] Connected to instance successfully.", instanceIp));
            instance.addSocket(socket);
            return socket;
        }
        catch (IOException e) {
            logger.debug(String.format("[%s] Socket connection failed! Trigger a refresh.", config.getCloudSqlInstance()));
            instance.forceRefresh();
            throw e;
        }
    }

    MonitoredCache getConnection(ConnectionConfig config) {
        ConnectionConfig updatedConfig = this.resolveConnectionName(config);
        MonitoredCache instance = this.instances.compute(updatedConfig, (k, v) -> v != null && !v.isClosed() ? v : new MonitoredCache(this.createConnectionInfo(updatedConfig), this.instanceNameResolverTimer, this::resolveDomain));
        instance.refreshIfExpired();
        return instance;
    }

    private ConnectionConfig resolveConnectionName(ConnectionConfig config) {
        if (Strings.isNullOrEmpty((String)config.getDomainName())) {
            return config;
        }
        if (!Strings.isNullOrEmpty((String)config.getCloudSqlInstance())) {
            return config.withDomainName(null);
        }
        try {
            CloudSqlInstanceName name = this.resolveDomain(config);
            return config.withCloudSqlInstance(name.getConnectionName());
        }
        catch (IllegalArgumentException e) {
            throw new IllegalArgumentException(String.format("Cloud SQL connection name is invalid: \"%s\"", config.getDomainName()), e);
        }
    }

    private CloudSqlInstanceName resolveDomain(ConnectionConfig config) {
        String unresolvedName = config.getDomainName();
        Function<String, String> resolver = config.getConnectorConfig().getInstanceNameResolver();
        CloudSqlInstanceName name = resolver != null ? this.instanceNameResolver.resolve(resolver.apply(unresolvedName)) : this.instanceNameResolver.resolve(unresolvedName);
        return name;
    }

    private ConnectionInfoCache createConnectionInfo(ConnectionConfig config) {
        logger.debug(String.format("[%s] Connection info added to cache.", config.getCloudSqlInstance()));
        if (config.getConnectorConfig().getRefreshStrategy() == RefreshStrategy.LAZY) {
            KeyPair keyPair = null;
            try {
                keyPair = (KeyPair)this.localKeyPair.get();
            }
            catch (InterruptedException | ExecutionException e) {
                throw new RuntimeException(e);
            }
            return new LazyRefreshConnectionInfoCache(config, this.adminApi, this.instanceCredentialFactory, keyPair);
        }
        return new RefreshAheadConnectionInfoCache(config, this.adminApi, this.instanceCredentialFactory, this.executor, this.localKeyPair, this.minRefreshDelayMs);
    }

    public void close() {
        logger.debug("Close all connections and remove them from cache.");
        this.instanceNameResolverTimer.cancel();
        this.instances.forEach((key, c) -> c.close());
        this.instances.clear();
    }
}

